/*****************************************************************************
/* J Drive (tm) Engine
/* Copyright 2000, Xilinx, Inc.
/* The use, reproduction, and distribution of this code is subject
/* to the J Drive license requirements and restrictions.
*****************************************************************************/
//
// File Name: bscanio.c
//
// Description: interface to the boundary scan hardware 
//
// Revision History:
//       Created:     Dec 2000
//       Last Update  July 10 2001 by Frank Phillips @ Xilinx
//
//-*************************************************************************

#include "bscanio.h"
#include "bscandrv.h"
#include "prgtyp.h"
#include  "sysopenv.h"
#include "logfkt.h"
#define SYSMSDelay(ms) delay(ms)
// Divide integer value with rounding up
#define IDIV(i1,i2) \
  ((i1 / i2) + ((i1 % i2) ? 1 : 0))

// Boundary Scan Controller States
char *vStateStrings[] = {
   "tapReset",
   "tapIdle",
   // DR
   "Select  DR",
   "Capture DR",
   "Shift   DR",
   "Exit1   DR",
   "Pause   DR",
   "Exit2   DR",
   "Update  DR",
   "Select  IR",
   "Capture IR",
   "Shift   IR",
   "Exit1   IR",
   "Pause   IR",
   "Exit2   IR",
   "Update  IR"
};

tBscanEngine BscanGraph[] =
  {
/* tapRESET     */  { tapReset, tapIdle },
/* tapIdle      */  { tapSelectDR,  tapIdle },
/* tapSELECTDR  */  { tapSelectIR,  tapCaptureDR },
/* tapCAPTUREDR */  { tapExit1DR,   tapShiftDR },
/* tapSHIFTDR   */  { tapExit1DR,   tapShiftDR },
/* tapEXIT1DR   */  { tapUpdateDR,  tapPauseDR },
/* tapPAUSEDR   */  { tapExit2DR,   tapPauseDR },
/* tapEXIT2DR   */  { tapUpdateDR,  tapShiftDR },
/* tapUPDATEDR  */  { tapSelectDR,  tapIdle },
/* tapSELECTIR  */  { tapReset, tapCaptureIR },
/* tapCAPTUREIR */  { tapExit1IR,   tapShiftIR },
/* tapSHIFTIR   */  { tapExit1IR,   tapShiftIR },
/* tapEXIT1IR   */  { tapUpdateIR,  tapPauseIR },
/* tapPAUSEIR   */  { tapExit2IR,   tapPauseIR },
/* tapEXIT2IR   */  { tapUpdateIR,  tapShiftIR },
/* tapUPDATEIR  */  { tapSelectDR,  tapIdle }
  };

// Interface to the Boundary Scan Input/Output Functions
// Delay for one milisecond
long  GetOneMsDelay(void);
// IEEE Std 1149.1
// Boundary Scan Interface Calls
//*************************************************
// This functions are called from executor
int  BScanCtrlInit( pBscanController aCtrl)
{int rt = rtErr;
  if (aCtrl !=NULL) {
    // Get the new Driver
    aCtrl->mDriver = NULL;
    aCtrl->mStatus = 0;
    aCtrl->mPortIndex = 0;
    aCtrl->mBscanGraph = BscanGraph;
    aCtrl->mDriver = JDriverNew();
    if (aCtrl->mDriver != NULL) {
    // loop to check the lpt Ports
    for (aCtrl->mPortIndex=0; aCtrl->mPortIndex<2; aCtrl->mPortIndex++) {
      aCtrl->mStatus = JDriverGetParallelPorts(
                              aCtrl->mDriver,
                              aCtrl->mPortIndex);
      // Error handling
      if (aCtrl->mStatus <= 0) {
        rt = rtErr;
      } else {
          rt = rtOk;
          // Check for XILINX Cable
          if (BScanCtrlCheckCable(aCtrl) == rtErr) {
            rt = rtErr;
            WRITEError("Cable not found !\n");
          };
          if (rt==rtOk) {
            InitBurstInfo(&aCtrl->mBurstInfo,tapBURSTSIZE);
        };
        return(rt);
      };
      }; // for
    };// Controller
  }
 // Error in findding driver
 if (rt == rtErr)
   WRITEError("Controller return error\n");
 return(rt);
};
// Destroy the boundary scan controller structure
void BScanCtrlClose( pBscanController aCtrl)
{
  if (aCtrl != NULL) {
//    JDriverDone(aCtrl->mDriver);
    JDriverClose(aCtrl->mDriver);
    JDriverFree(aCtrl->mDriver);
    DoneBurstInfo(&aCtrl->mBurstInfo);
  };
};
// read one byte back from hardware
tByte BScanCtrlReadByte(pBscanController aCtrl)
{
  return(JDriverReadByte(aCtrl->mDriver,GETStatusPort(aCtrl->mDriver)));
};
void  BScanCtrlWriteByte(pBscanController aCtrl,tByte aByte)
{ JDriverWriteByte(aCtrl->mDriver,
                   GETDataPort(aCtrl->mDriver),
                   aByte);
};
// Execute the burst - that means put data to hardware
// and get the data from hardware in one step
int   BScanCtrlBurstExecute(pBscanController aCtrl, int aReadBack)
{ pBurstInfo coBuInf = &aCtrl->mBurstInfo;
  int rt = rtErr;
  int idx;
  tByte vTdi;
  tByte vTdo;
  long vDrvSize = (aCtrl->mHaltState?2*coBuInf->mCount:3*coBuInf->mCount);
  long vStep    = (aCtrl->mHaltState?2:3);
  long vTdoIdx  = 0L;
  long vIdx     = 0L; // Input buffer Index
    // execute the burst
   InitDrvBuffer(coBuInf,vDrvSize);
   coBuInf->mDrvCount = 0L;
   for (idx = 0; idx<coBuInf->mCount;idx++)
   {
    vTdi = coBuInf->mTdi[idx];
    // TCK-Low
    vTdi = vTdi & ~ bsdrvTCK;
//    BScanCtrlWriteByte(aCtrl, vTdi);
    coBuInf->mDrvOutput[coBuInf->mDrvCount]=vTdi;coBuInf->mDrvCount++;
    // TCK-High
    vTdi |= bsdrvTCK;
//    BScanCtrlWriteByte(aCtrl, vTdi);
      coBuInf->mDrvOutput[coBuInf->mDrvCount]=vTdi;coBuInf->mDrvCount++;

    // If one Device required TCK-Low in halt-state give a TCK-Low
    if (!aCtrl->mHaltState) {
      vTdi = vTdi & ~ bsdrvTCK;
//      BScanCtrlWriteByte(aCtrl, vTdi);
    coBuInf->mDrvOutput[coBuInf->mDrvCount]=vTdi;coBuInf->mDrvCount++;
    };
   }; // for
//   if (JDriverWRTransfer(aCtrl->mDriver, coBuInf->mDrvOutput,coBuInf->mDrvInput,coBuInf->mDrvCount));
/* Semicolon was taken out from the above line. 
Frank Phillips @ Xilinx
Jul_10_2001
*/
   if (JDriverWRTransfer(aCtrl->mDriver, coBuInf->mDrvOutput,coBuInf->mDrvInput,coBuInf->mDrvCount))
       rt = rtOk;
   // Get the data from driver if required
   if (aReadBack)
   for (idx = 0; idx < coBuInf->mDrvCount;idx++) {
     vTdoIdx++;
     // 2 Bytes are write then read back the tdo
     if (vTdoIdx == 2) {
       vTdo = coBuInf->mDrvInput[idx]& bsdrvTDO;
       coBuInf->mTdo[vIdx] = vTdo;
       vIdx++;
       if (vTdoIdx == vStep) vTdoIdx = 0L;
     }; // if
   };// for
  return(rt);
};
//*************************************************
// Checks the presents of XILINX cable
#define ACKLow  0x10
#define ACKHigh 0x50
int   BScanCtrlCheckCable(pBscanController aCtrl)
{tByte  vByteLow;
  tByte vByteHigh;
 int   rt = rtErr;
   if (aCtrl!=NULL) {
   // Writeing Check Bytes
       BScanCtrlWriteByte(aCtrl,ACKLow);
       vByteLow = BScanCtrlReadByte(aCtrl);
       BScanCtrlWriteByte(aCtrl,ACKHigh);
       vByteHigh = BScanCtrlReadByte(aCtrl);
   // Reading back Checkbytes
      if (vByteLow != vByteHigh)
          rt = rtOk;
   };
   return(rt);
};
// Write Clocks to the hardware
void WriteClocks(pBscanController aCtrl, long aCount)
{ int idx;

  for (idx = 0; idx< aCount;idx++) {
    BScanCtrlWriteByte(aCtrl,0x00| bsdrvOne);
    BScanCtrlWriteByte(aCtrl,bsdrvTCK | bsdrvOne);
    };
};
// ************** delay functions ********************
//Makes a delay for aLoopCount loops
// depending on buffer size
void Delays(long aCount)
{ double vCalc;
  while (aCount != 0L) {
    aCount--;
  };
};
// Wait for time of Micro seconds


// Boundary Scan Engine goes to the next state
int GotoNextState(pBscanController aCtrl, int aTms)
{ int rt = rtErr;
  if (aCtrl != NULL) {
    aCtrl->mTapState =(aTms?
      aCtrl->mBscanGraph[aCtrl->mTapState].mTMSHigh
        :aCtrl->mBscanGraph[aCtrl->mTapState].mTMSLow);
  rt = rtOk;
  };
  return(rt);
};

//****************************************************************************
#define mnMeasCount 5

void InitDelay(pBscanController aCtrl)
{
    int TickCountStart, TickCountEnd;
    int LoopCnt;
    int MeasNr;
    int Ticks[mnMeasCount];
    int Calls[mnMeasCount];
    double CallsPerMS[mnMeasCount];

    TickCountEnd = TickCountStart = GetTickCount();
    while (TickCountEnd == TickCountStart)
        TickCountEnd = GetTickCount();
    for (MeasNr = 0; MeasNr < mnMeasCount; MeasNr ++)
    {
        TickCountStart = TickCountEnd;
        for(LoopCnt = 0;TickCountEnd == TickCountStart; LoopCnt++)
        {
            TickCountEnd = GetTickCount();
        }
        Ticks[MeasNr]  = TickCountEnd - TickCountStart;
        Calls[MeasNr] = LoopCnt;
    }
/* The start val of MeasNr was change to 0 from 1 in the below line
Frank Phillips @ Xilinx
Jul_10_2001
*/
    for (MeasNr = 0; MeasNr < mnMeasCount; MeasNr ++)
        CallsPerMS[MeasNr] = ((double)Calls[MeasNr]) / Ticks[MeasNr];

    aCtrl->mMaxCallsPerMS = CallsPerMS[0];
    for (MeasNr = 1; MeasNr < mnMeasCount; MeasNr ++)
    {
        if (CallsPerMS[MeasNr] > aCtrl->mMaxCallsPerMS)
        {
            aCtrl->mMaxCallsPerMS = CallsPerMS[MeasNr];
        }
    }

}
void DelaySeconds(pBscanController aCtrl, double aSeconds)
{
    int LoopCnt;
    int TickCountEnd;
    int LoopCntEnd;

    LoopCntEnd = aCtrl->mMaxCallsPerMS;
    aSeconds = aSeconds * 1000; // -> milliseconds
    for(; aSeconds > 1; aSeconds -= 1)
        for (LoopCnt = 0; LoopCnt < LoopCntEnd; LoopCnt++)
        {
            TickCountEnd = GetTickCount();
//          TickCountEnd++;
        }
    LoopCntEnd = aSeconds * aCtrl->mMaxCallsPerMS + 1;
    for (LoopCnt = 0; LoopCnt < LoopCntEnd; LoopCnt++)
    {
        TickCountEnd = GetTickCount();
    }

}

//****************************************************************************

