/*****************************************************************************
/* J Drive (tm) Engine
/* Copyright 2000, Xilinx, Inc.
/* The use, reproduction, and distribution of this code is subject
/* to the J Drive license requirements and restrictions.
*****************************************************************************/
#include "libenv.h"
#include "prgdev.h"
#include "gbsd_tab.h"

#define PARSERControl \
    ((struct tParserControl *) ParserControl)

#define READCharacter(c) \
    { \
        READCharBsdlFile(c, PARSERControl->mBsdlFile, vEof); \
        if (c == '\n') \
        { \
            PARSERControl->mLine++; \
            STRDone(PARSERControl->mLineBuf); \
            STRInit(PARSERControl->mLineBuf, ""); \
        } \
        else \
            STRAddChar(PARSERControl->mLineBuf, c); \
    }

#define UNREADCharacter(c) \
    { \
        UNREADCharBsdlFile(c, PARSERControl->mBsdlFile); \
    STRRemoveLastChar(PARSERControl->mLineBuf); \
        if (c == '\n') \
            PARSERControl->mLine--; \
    }

/* Added by Frank Phillips @ Xilinx to remove memory leak on
Jun_06_2001 */
#define FREEParserControl \
        { \
                STRDone(vParserControl.mStrBuf); \
                STRDone(vParserControl.mLineBuf); \
                FreeDataExpr(vParserControl.mDataExpr); \
                FreeDurationSpec(vParserControl.mDurationSpec); \
        }

#define STRToStack \
{ \
  STRInit(vStrBuf, STRPChar(PARSERControl->mStrBuf)); \
  AddListItem(&vStringBuffer, vStrBuf.mString); \
  lvalp->mTString = vStrBuf; \
}

struct tParserControl
{
    int mReadReal;                              // Enables token REAL
    int mReadChar;                              // Enables token CHAR
    int mReadBitMask;                           // Enables token BIT_MASK
    int mReadHex;                                   // Enables token HEX
    int mReadString;                            // Enables token VHDL_STRING
    int mQuotes;                                    // Indicates open string
    int mLine;                                      // Common line in parsing
    int mError;                                     // Indicates semantic error
    int mEmpty;                                     // Error if bsdl file is empty
    TSTRING mStrBuf;                            // String buffer
    TSTRING mLineBuf;                           // String buffer (common line)
    TFILEBsdl mBsdlFile;                    // Bsdl file
    pDevEnt mDevice;                            // Device
    int mRegLength;                             // Register length (register access)
    pDataExpr mDataExpr;                    // Data expression
    int mCapFieldWidth;                     // Capture field width
    int mOutputOp;                              // Indicates output operator
    pDurationSpec mDurationSpec;    // Duration specification
    int mExclusivity;                           // Indicates action exclusivity option
    int mActionOpt;                             // Action mandatory, OPTIONAL or RECOMMENDED
};

/* Definitions added by Frank Phillips @ Xilinx on
06_05_2001
*/

int BSDLparse(void *);
static char OverreadComment(struct tParserControl *);
static char OverreadSringConcat(struct tParserControl *);
int getAttribute(YYSTYPE *lvalp, struct tParserControl * ParserControl);
tList vStringBuffer;

/* -------------------------------------------------------------------------- */
/* --- inits parser control and starts file parsing                       --- */
/* -------------------------------------------------------------------------- */
int parseBsdlFile(pDevEnt aDevice)
{
    struct tParserControl vParserControl;
    char *vFileName = NULL, tmp_pchar;
    int vError = rtOk, i ;

    // init
    vParserControl.mReadReal = 0;
    vParserControl.mReadChar = 0;
    vParserControl.mReadBitMask = 0;
    vParserControl.mReadHex = 0;
    vParserControl.mReadString = 0;
    vParserControl.mQuotes = 0;
    vParserControl.mLine = 1;
    vParserControl.mError = rtOk;
  vParserControl.mEmpty = 0;
    STRInit(vParserControl.mStrBuf, "");
    STRInit(vParserControl.mLineBuf, "");
    vParserControl.mDevice = aDevice;
  vParserControl.mRegLength = 0;
    vParserControl.mDataExpr = InitDataExpr();
    vParserControl.mCapFieldWidth = 0;
    vParserControl.mOutputOp = fopNoOutput;
    vParserControl.mDurationSpec = InitDurationSpec();
    vParserControl.mExclusivity = fNoExclOpt;
    vParserControl.mActionOpt = fMandAct;

    if (STRLen(BSDLFileNameOfDevice(vParserControl.mDevice)) == 0)
    {
         FREEParserControl;
         return rtOk;
    }

    vFileName = STRPChar(BSDLFileNameOfDevice(vParserControl.mDevice));
    if (vFileName == NULL)
    {
        FREEParserControl;
        return rtOk;
    }
    // open bsdl file
    OPENBsdlFile(vParserControl.mBsdlFile, vFileName);
    if (vParserControl.mBsdlFile == NULL)
    {
        WRITEError("...File \"%s\" not found (device %s).\n", vFileName,
            STRPChar(DEVICEName(vParserControl.mDevice)));
        FREEParserControl;
        return rtErr;
    }

/* The following two functioncal calls were added by
by Frank Phillips @ Xilinx to remove memory leak on
Jun_06_2001 */
    InitList(&vStringBuffer, loptFREE);
   //	InitList(&vStringBuffer);
    SetListFunc(&vStringBuffer, NULL, NULL, (pfFreeItem)free );

    // parse (fill bsdl info)
    WRITEStatus("...Start parsing bsdl file \"%s\" (device %s).\n", vFileName,
        STRPChar(DEVICEName(vParserControl.mDevice)));
    if (BSDLparse((void *) &vParserControl))
    {
        WRITEError("\n    ...line:%d>%s\n", vParserControl.mLine,
            STRPChar(vParserControl.mLineBuf));
        vError = rtErr;
    }
    CLOSEBsdlFile(vParserControl.mBsdlFile);

    DoneList(&vStringBuffer);
    FREEParserControl;
//    STRDone(vParserControl.mStrBuf);
//    STRDone(vParserControl.mLineBuf);

/* The following three  functioncal calls were added by
by Frank Phillips @ Xilinx to remove memory leak on
Jun_06_2001 */

/*
     for (i=0; i < ITEMCountOfList((&vStringBuffer)); i++)
     {
		tmp_pchar = GetItemAtIndex(&vStringBuffer, i);
        FREEMem(GetItemAtIndex(&vStringBuffer, i), NULL);
     }
     DoneList(&vStringBuffer);
*/
//   FreeDataExpr(vParserControl.mDataExpr);
// FreeDurationSpec(vParserControl.mDurationSpec);


    // return error if unsuccessfully parsing
    if (vParserControl.mEmpty)
        vError |= rtErr;
    return vError;
}

/* -------------------------------------------------------------------------- */
/* --- parse error handling                                               --- */
/* -------------------------------------------------------------------------- */
int BSDLerror(char *aErrorString)
{
    WRITEError("  ...parse error:");
    return 0;
}

/* -------------------------------------------------------------------------- */
/* --- scanner - get the tokens for parser                                --- */
/* -------------------------------------------------------------------------- */
int BSDLlex(YYSTYPE *lvalp, int ParserControl)
{
    char c, ci;
    int i, vStrLen;
    int vIsAlpha, vIsHex, vIsBitMask , vIsDigit;
    int vEof;
    TSTRING vStrBuf;

    STRDone(PARSERControl->mStrBuf);


    // Ignore whitespace, get first nonwhite character.
    READCharacter(c);
    while (c == ' ' || c == '\t' || c == '\n')
        READCharacter(c);

    // Ignore Comments
    if (c == '-')
        c = OverreadComment((struct tParserControl *) ParserControl);

    // Concat strings (Overread -"&"-)
    if ((c == '"') && PARSERControl->mQuotes)
    {
        c = OverreadSringConcat(PARSERControl);
        while (c == ' ' || c == '\t' || c == '\n')
            READCharacter(c);
    }
    if (c == '"')
        PARSERControl->mQuotes = ! PARSERControl->mQuotes;

    // Overread strings (-> description warning)
    if (PARSERControl->mReadString)
    {
        do
        {
            // Add character to the buffer.
            STRAddChar(PARSERControl->mStrBuf, c);
            // Get another character.
            READCharacter(c);
            // Concat strings (Overread -"&"-)
            if ((c == '"') && PARSERControl->mQuotes)
                c = OverreadSringConcat(PARSERControl);
        }
        while ((c != '"') && (! vEof));

        UNREADCharacter(c);
        //STRInit((lvalp->mTString), STRPChar(PARSERControl->mStrBuf));
        //AddListItem(&vStringBuffer, lvalp->mTString.mString);
        return VHDL_STRING;
    }

    // End of file
    if (vEof)
        return 0;

    // Read the string.
    if (CHARIsAlnum(c) || (c == '$'))
    {
        do
        {
            // Add this character to the buffer.
            STRAddChar(PARSERControl->mStrBuf, c);
            // Get another character.
            READCharacter(c);
            // Concat strings (Overread -"&"-)
            if ((c == '"') && PARSERControl->mQuotes)
                c = OverreadSringConcat(PARSERControl);
        }
        while (c != vEof && (CHARIsAlnum (c) || (c == '_')));

        UNREADCharacter(c);
        // STRInit((lvalp->mTString), STRPChar(PARSERControl->mStrBuf));
        // AddListItem(&vStringBuffer, lvalp->mTString.mString);
        STRToStack;

        // get the token
        // ATTRIBUTE
        if (STRChEqualNoCase(PARSERControl->mStrBuf, "attribute"))
            return getAttribute(lvalp, (struct tParserControl *) ParserControl);

        // general
        if (STRChEqualNoCase(PARSERControl->mStrBuf, "generic"))
            return GENERIC;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "entity"))
            return ENTITY;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "is"))
            return IS;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "of"))
            return OF;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "string"))
            return STRING;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "end"))
            return END;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "use"))
            return USE;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "all"))
            return ALL;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "true"))
            return TRUE;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "false"))
            return FALSE;
        // standard packages
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "STD_1149_1_1990"))
            return STD_1149_1_1990;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "STD_1149_1_1993"))
            return STD_1149_1_1993;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "STD_1149_1_1994"))
            return STD_1149_1_1994;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "STD_1532_2000"))
            return STD_1532_2000;
        // pin mapping
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "PHYSICAL_PIN_MAP"))
            return PHYSICAL_PIN_MAP;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "port"))
            return PORT;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "in"))
            return IN;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "out"))
            return OUT;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "buffer"))
            return BUFFER;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "inout"))
            return INOUT;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "linkage"))
            return LINKAGE;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "bit"))
            return BIT;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "bit_vector"))
            return BIT_VECTOR;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "to"))
            return TO;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "downto"))
            return DOWNTO;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "constant"))
            return CONSTANT;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "PIN_MAP_STRING"))
            return PIN_MAP_STRING;
        // grouped port identification
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "DIFFERENTIAL_VOLTAGE"))
            return DIFFERENTIAL_VOLTAGE;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "DIFFERENTIAL_CURRENT"))
            return DIFFERENTIAL_CURRENT;
        // scan port identification
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "signal"))
            return SIGNAL;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "LOW"))
            return LOW;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "BOTH"))
            return BOTH;
        // register access description
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "CAPTURES"))
            return CAPTURES;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "BOUNDARY"))
            return BOUNDARY;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "BYPASS"))
            return BYPASS;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "DEVICE_ID"))
            return DEVICE_ID;
        // boundary scan register description
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "input"))
            return INPUT;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "output2"))
            return OUTPUT2;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "output3"))
            return OUTPUT3;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "control"))
            return CONTROL;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "controlr"))
            return CONTROLR;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "internal"))
            return INTERNAL;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "clock"))
            return CLOCK;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "bidir"))
            return BIDIR;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "observe_only"))
            return OBSERVE_ONLY;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "weak0"))
            return WEAK0;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "weak1"))
            return WEAK1;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "pull0"))
            return PULL0;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "pull1"))
            return PULL1;
        // runbist description
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "wait_duration"))
            return WAIT_DURATION;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "observing"))
            return OBSERVING;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "at_pins"))
            return AT_PINS;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "expect_data"))
            return EXPECT_DATA;

        // bsdl extension STD_1532_2000
        // isc pin behavior
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "pins"))
            return PINS;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "highz"))
            return HIGHZ;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "clamp"))
            return CLAMP;
        // isc status
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "not"))
            return NOT;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "implemented"))
            return IMPLEMENTED;
        // isc protection
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "ISC_DISABLE_READ"))
            return ISC_DISABLE_READ;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "ISC_DISABLE_PROGRAM"))
            return ISC_DISABLE_PROGRAM;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "ISC_DISABLE_ERASE"))
            return ISC_DISABLE_ERASE;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "ISC_DISABLE_KEY"))
            return ISC_DISABLE_KEY;
        // isc flow
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "Initialize"))
            return INITIALIZE;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "Repeat"))
            return REPEAT;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "Terminate"))
            return TERMINATE;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "wait"))
            return WAIT;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "min"))
            return MIN;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "max"))
            return MAX;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "CRC"))
            return CRC;
        // isc action
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "proprietary"))
            return PROPRIETARY;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "optional"))
            return OPTIONAL;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "recommended"))
            return RECOMMENDED;

        // VARIABLE
        if (STRPChar(PARSERControl->mStrBuf)[0] == '$')
            return VARIABLE;

        // id, hex, real, integer, bit or bit_mask ?
        vIsHex = 1;
        vIsAlpha = 0;
        vIsBitMask = 1;
        vStrLen = STRLen(PARSERControl->mStrBuf);

        for (i = 0; i < vStrLen; i++)
        {
            ci = STRPChar(PARSERControl->mStrBuf)[i];
            if ((ci < '0') || ((ci > '9') && (ci < 'A')) || ((ci > 'F')
                    && (ci < 'a')) || (ci > 'f') || (c == '_'))
                vIsHex = 0;
            if (((ci >= 'A') && (ci <= 'Z')) || ((ci >= 'a') && (ci <= 'z'))
                    || (c == '-'))
                vIsAlpha = 1;
            if ((ci != '0') && (ci != '1') && (ci != 'x') && (ci != 'X'))
                vIsBitMask = 0;
        }

        // REAL
        if (PARSERControl->mReadReal && (! vIsAlpha) && (c == '.'))
        {
            // 0.
            READCharacter(c);
            STRAddChar(PARSERControl->mStrBuf, c);
            READCharacter(c);

            vIsDigit = 0;
            while (CHARIsDigit(c))
            {
                vIsDigit = 1;
                STRAddChar(PARSERControl->mStrBuf, c);
                READCharacter(c);
            }
            // 0.0 --> REAL
            if ((! CHARIsAlpha(c)) && (c != '_') && (c != '.') && vIsDigit)
            {
                UNREADCharacter(c);
                // STRInit((lvalp->mTString), STRPChar(PARSERControl->mStrBuf));
                // AddListItem(&vStringBuffer, lvalp->mTString.mString);
                STRToStack;
                return REAL;
            }
            // 0.0e
            else if ((c == 'e') && vIsDigit)
            {
                STRAddChar(PARSERControl->mStrBuf, c);
                READCharacter(c);

                // 0.0e+-
                if ((c == '+') || (c == '-'))
                {
                    STRAddChar(PARSERControl->mStrBuf, c);
                    READCharacter(c);
                    vIsDigit = 0;

                    while (CHARIsDigit(c))
                    {
                        vIsDigit = 1;
                        STRAddChar(PARSERControl->mStrBuf, c);
                        READCharacter(c);
                    }
                    // 0.0e+-0 --> REAL
                    if ((! CHARIsAlpha(c)) && (c != '_') && (c != '.') && vIsDigit)
                    {
                        UNREADCharacter(c);
                        // STRInit((lvalp->mTString), STRPChar(PARSERControl->mStrBuf));
                        // AddListItem(&vStringBuffer, lvalp->mTString.mString);
                        STRToStack;
                        return REAL;
                    }
                    // 0.0e+-x   --> INVALID
                    else
                        return INVALID;
                }
                // 0.0e0
                else if (CHARIsDigit(c))
                {
                    STRAddChar(PARSERControl->mStrBuf, c);
                    READCharacter(c);

                    vIsDigit = 0;
                    while (CHARIsDigit(c))
                    {
                        STRAddChar(PARSERControl->mStrBuf, c);
                        READCharacter(c);
                    }
                    // 0.0e0 --> REAL
                    if ((! CHARIsAlpha(c)) && (c != '_') && (c != '.'))
                    {
                        UNREADCharacter(c);
                        // STRInit((lvalp->mTString), STRPChar(PARSERControl->mStrBuf));
                        // AddListItem(&vStringBuffer, lvalp->mTString.mString);
                        STRToStack;
                        return REAL;
                    }
                    // 0.0e0x   --> INVALID
                    else
                        return INVALID;
                }
                // 0.0ex --> INVALID
                else
                    return INVALID;
            }
            // 0.0x or 0.x --> INVALID
            else
                return INVALID;
        }

        // BIT_MASK
        if (PARSERControl->mReadBitMask && vIsBitMask)
        {
            if (! vIsAlpha)
            {
                if (vStrLen == 32)
                    return BIT_MASK_32_01;
                return BIT_MASK_01;
            }
            if (vStrLen == 32)
                return BIT_MASK_32;
            return BIT_MASK;
        }

        // CHAR (in grammar: 0, 1, X or Z)
        if (PARSERControl->mReadChar && (STRLen(PARSERControl->mStrBuf) == 1))
            return STRPChar(PARSERControl->mStrBuf)[0];

        // INTEGER
        if (! vIsAlpha)
            return INTEGER;

        // HEX
        if (PARSERControl->mReadHex && vIsHex)
            return HEX;

        // VHDL_ID
        if (CHARIsAlpha(STRPChar(PARSERControl->mStrBuf)[0]))
            return VHDL_ID;

        // Invalid string
        if (vStrLen > 1)
            return INVALID;
    }
    // Any other character is a token by itself.
    return c;
}

/* -------------------------------------------------------------------------- */
/* --- scanner - get the attribute tokens for parser                      --- */
/* -------------------------------------------------------------------------- */
int getAttribute(YYSTYPE *lvalp, struct tParserControl * ParserControl)
{
    char c; int vEof;
    TSTRING vStrBuf;

    STRDone(PARSERControl->mStrBuf);

    // Ignore whitespace, get first nonwhite character.
    READCharacter(c);
    while (c == ' ' || c == '\t' || c == '\n')
        READCharacter(c);

    // End of file
    if (vEof)
        return 0;

    // Char starts an identifier => read the name.
    if (CHARIsAlpha(c))
    {
        do
        {
            // Add character to the buffer.
            STRAddChar(PARSERControl->mStrBuf, c);
            // Get another character.
            READCharacter(c);
        }
        while ((! vEof) && (CHARIsAlnum(c) || (c == '_')));

        UNREADCharacter(c);
        // STRInit((lvalp->mTString), STRPChar(PARSERControl->mStrBuf));
        // AddListItem(&vStringBuffer, lvalp->mTString.mString);
        STRToStack;

        // get the token
        // component conformance statement
        if (STRChEqualNoCase(PARSERControl->mStrBuf, "COMPONENT_CONFORMANCE"))
            return ATTRIBUTE_COMPONENT_CONFORMANCE;
        // pin mapping
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "PIN_MAP"))
            return ATTRIBUTE_PIN_MAP;
        // grouped port identification
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "PORT_GROUPING"))
            return ATTRIBUTE_PORT_GROUPING;
        // scan port identification
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "TAP_SCAN_CLOCK"))
            return ATTRIBUTE_TAP_SCAN_CLOCK;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "TAP_SCAN_IN"))
            return ATTRIBUTE_TAP_SCAN_IN;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "TAP_SCAN_OUT"))
            return ATTRIBUTE_TAP_SCAN_OUT;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "TAP_SCAN_MODE"))
            return ATTRIBUTE_TAP_SCAN_MODE;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "TAP_SCAN_RESET"))
            return ATTRIBUTE_TAP_SCAN_RESET;
        // compliance enable description
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "COMPLIANCE_PATTERNS"))
            return ATTRIBUTE_COMPLIANCE_PATTERNS;
        // instruction register description
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "INSTRUCTION_LENGTH"))
            return ATTRIBUTE_INSTRUCTION_LENGTH;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "INSTRUCTION_OPCODE"))
            return ATTRIBUTE_INSTRUCTION_OPCODE;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "INSTRUCTION_CAPTURE"))
            return ATTRIBUTE_INSTRUCTION_CAPTURE;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "INSTRUCTION_PRIVATE"))
            return ATTRIBUTE_INSTRUCTION_PRIVATE;
        // optional register description
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "IDCODE_REGISTER"))
            return ATTRIBUTE_IDCODE_REGISTER;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "USERCODE_REGISTER"))
            return ATTRIBUTE_USERCODE_REGISTER;
        // register access description
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "REGISTER_ACCESS"))
            return ATTRIBUTE_REGISTER_ACCESS;
        // boundary scan register description
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "BOUNDARY_LENGTH"))
            return ATTRIBUTE_BOUNDARY_LENGTH;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "BOUNDARY_REGISTER"))
            return ATTRIBUTE_BOUNDARY_REGISTER;
        // runbist description
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "RUNBIST_EXECUTION"))
            return ATTRIBUTE_RUNBIST_EXECUTION;
        // intest description
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "INTEST_EXECUTION"))
            return ATTRIBUTE_INTEST_EXECUTION;
        // design warning
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "DESIGN_WARNING"))
            return ATTRIBUTE_DESIGN_WARNING;

        // bsdl extension STD_1532_2000
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "ISC_STATUS"))
            return ATTRIBUTE_ISC_STATUS;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "ISC_PIN_BEHAVIOR"))
            return ATTRIBUTE_ISC_PIN_BEHAVIOR;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "ISC_BLANK_USERCODE"))
            return ATTRIBUTE_ISC_BLANK_USERCODE;
/* 
It is now changed to ISC_SECURITY
with Dec 10 2000 revision
Modifed by frank.phillips@xilinx.com
Date: Feb-12-01
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "ISC_PROTECTION"))
            return ATTRIBUTE_ISC_PROTECTION;
*/
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "ISC_SECURITY"))
            return ATTRIBUTE_ISC_SECURITY;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "ISC_FLOW"))
            return ATTRIBUTE_ISC_FLOW;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "ISC_PROCEDURE"))
            return ATTRIBUTE_ISC_PROCEDURE;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "ISC_ACTION"))
            return ATTRIBUTE_ISC_ACTION;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "ISC_ILLEGAL_EXIT"))
            return ATTRIBUTE_ISC_ILLEGAL_EXIT;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "ISC_DESIGN_WARNING"))
            return ATTRIBUTE_ISC_DESIGN_WARNING;
    }
    // unknown attribute (no token)
    return ATTRIBUTE_UNKNOWN;
}

/* -------------------------------------------------------------------------- */
/* --- ignores comments                                                   --- */
/* -------------------------------------------------------------------------- */
static char OverreadComment(struct tParserControl *ParserControl)
{
    char c = '-';
    int vEof;

    while (c == '-')
    {
        // Overread multiple comments
        READCharacter(c);
        if (c == '-')
        {
            // Overread comment
            while (c != '\n')
                READCharacter(c);
            while (c == ' ' || c == '\t' || c == '\n')
                READCharacter(c);
        }
        else
        {
            // No comment -> return token '-'
            UNREADCharacter(c);
            return '-';
        }
    }
    return c;
}

/* -------------------------------------------------------------------------- */
/* --- concat bsdl strings (ignores -"&"-)                                --- */
/* -------------------------------------------------------------------------- */
static char OverreadSringConcat(struct tParserControl *ParserControl)
{
    char c;
    int vEof;

    // Ignore whitespace, get first nonwhite character.
    READCharacter(c);
    while (c == ' ' || c == '\t' || c == '\n')
        READCharacter(c);

    // Ignore Comments
    if (c == '-')
        c = OverreadComment(ParserControl);
    while (c == ' ' || c == '\t' || c == '\n')
        READCharacter(c);

    // No string concatenation
    if (c != '&')
    {
        UNREADCharacter(c);
        return '"';
    }

    // Ignore whitespace, get first nonwhite character.
    READCharacter(c);
    while (c == ' ' || c == '\t' || c == '\n')
        READCharacter(c);

    // Ignore Comments
    if (c == '-')
        c = OverreadComment(ParserControl);
    while (c == ' ' || c == '\t' || c == '\n')
        READCharacter(c);

    // No string concatenation
    if (c != '"')
    {
        UNREADCharacter(c);
        UNREADCharacter('&');
        return '"';
    }

    // String concatenation overread
    READCharacter(c);
    return c;
}




