/*****************************************************************************
/* J Drive (tm) Engine
/* Copyright 2000, Xilinx, Inc.
/* The use, reproduction, and distribution of this code is subject
/* to the J Drive license requirements and restrictions.
*****************************************************************************/
#include "libenv.h"
#include "progty.h"
#include "gcmd_tab.h"

#define PARSERControl \
    ((struct tParserControl *) ParserControl)

#define READCharacter(c) \
    { \
        READCharCommandFile(c, PARSERControl->mCommandFile, vEof); \
        if (c == '\n') \
        { \
            PARSERControl->mLine++; \
            STRDone(PARSERControl->mLineBuf); \
            STRInit(PARSERControl->mLineBuf, ""); \
        } \
        else \
            STRAddChar(PARSERControl->mLineBuf, c); \
    }

#define UNREADCharacter(c) \
    { \
        UNREADCharCommandFile(c, PARSERControl->mCommandFile); \
    STRRemoveLastChar(PARSERControl->mLineBuf); \
        if (c == '\n') \
            PARSERControl->mLine--; \
    }

struct tParserControl
{
    int mReadStringId;                      // Enables token STRING_ID
    int mLine;                                      // Common line in parsing
    int mEmpty;                                     // Error if bsdl file is empty
    TSTRING mStrBuf;              // String buffer
    TSTRING mLineBuf;                           // String buffer (common line)
    TFILECommand mCommandFile;    // Command file
    pProg1532 mDeviceList;              // Device list
};

/* -------------------------------------------------------------------------- */
/* --- inits parser control and starts file parsing                       --- */
/* -------------------------------------------------------------------------- */
int parseCommandFile(pProg1532 aDeviceList, char *aFileName)
{
    struct tParserControl vParserControl;
    int vError = rtOk;

    // init
    vParserControl.mReadStringId = 0;
    vParserControl.mLine = 1;
  vParserControl.mEmpty = 0;
    STRInit(vParserControl.mStrBuf, "");
    STRInit(vParserControl.mLineBuf, "");
    vParserControl.mDeviceList = aDeviceList;

    // open command file
    OPENCommandFile(vParserControl.mCommandFile, aFileName);
  if (vParserControl.mCommandFile == NULL)
    {
        WRITEError("...File \"%s\" not found.\n", aFileName);
        return rtErr;
    }
  // parse (fill scan chain structure)
    WRITEStatus("...Start parsing command file \"%s\".\n", aFileName);
    if (CMDparse((void *) &vParserControl))
    {
        WRITEError("\n    ...line:%d>%s\n", vParserControl.mLine,
            STRPChar(vParserControl.mLineBuf));
        vError = rtErr;
    }
    CLOSECommandFile(vParserControl.mCommandFile);

  STRDone(vParserControl.mStrBuf);
    STRDone(vParserControl.mLineBuf);

    // return error if unsuccessfully parsing
    if (vParserControl.mEmpty)
        vError |= rtErr;
    return vError;
}

/* -------------------------------------------------------------------------- */
/* --- parse error handling                                               --- */
/* -------------------------------------------------------------------------- */
int CMDerror (aErrorString)
{
    WRITEError("  ...parse error:");
    return 0;
}

/* -------------------------------------------------------------------------- */
/* --- scanner - get the tokens for parser                                --- */
/* -------------------------------------------------------------------------- */
int CMDlex(lvalp, ParserControl)
    YYSTYPE *lvalp;
{
    char c, ci;
    int i, vStrLen;
    int vIsHex, vIsAlpha;
    int vEof;

    STRDone(PARSERControl->mStrBuf);

    // Ignore whitespace, get first nonwhite character./
    READCharacter(c);
    while (c == ' ' || c == '\t' || c == '\n')
        READCharacter(c);

    // End of file/
    if (vEof)
        return 0;

    // Get the token (an option)
    if (c == '-')
    {
        READCharacter(c);

        if (c == 'i')
            return INSTRUCTION;
        else if (c == 'b')
            return BSDL_FILE;
        else if (c == 'a')
            return ACTION;
        else if (c == 'd')
            return DATA_FILE;
        else if (c == 'o')
            return OUTPUT_DATA_FILE;
        // invalid option
        else
            return '-';
    }

    // STRING_ID
    if (PARSERControl->mReadStringId)
    {
        // empty string
        if (c == '"')
            return INVALID;

        // read string
        do
        {
            // Add character to the buffer.
            STRAddChar(PARSERControl->mStrBuf, c);
            // Get another character.
            READCharacter(c);
        }
        while ((c != '\n') && (c != '"'));

        // Only characters, digits, whitespace and underline allow
        if (c != '"')
            return INVALID;

        UNREADCharacter(c);
        STRInit((lvalp->mTString), STRPChar(PARSERControl->mStrBuf));
        return STRING_ID;
    }

    // Read the string.
    if (CHARIsAlnum(c))
    {
        do
        {
            // Add this character to the buffer.
            STRAddChar(PARSERControl->mStrBuf, c);
            // Get another character.
            READCharacter(c);
        }
        while (c != vEof && (CHARIsAlnum (c) || (c == '_')));

        UNREADCharacter(c);
        STRInit((lvalp->mTString), STRPChar(PARSERControl->mStrBuf));

        // get the token
        // id or integer
        vIsAlpha = 0;
        vStrLen = STRLen(PARSERControl->mStrBuf);

    for (i = 0; i < vStrLen; i++)
        {
            ci = STRPChar(PARSERControl->mStrBuf)[i];
            if (((ci >= 'A') && (ci <= 'Z')) || ((ci >= 'a') && (ci <= 'z'))
                    || (c == '-'))
                vIsAlpha = 1;
        }

        // INTEGER
        if (! vIsAlpha)
            return INTEGER;

        // VHDL_ID
        if (CHARIsAlpha(STRPChar(PARSERControl->mStrBuf)[0]))
            return VHDL_ID;

        // Invalid string
        if (vStrLen > 1)
            return INVALID;
    }
    // Any other character is a token by itself.
    return c;
}

