/*****************************************************************************
/* J Drive (tm) Engine
/* Copyright 2000, Xilinx, Inc.
/* The use, reproduction, and distribution of this code is subject
/* to the J Drive license requirements and restrictions.
*****************************************************************************/
#include "libenv.h"
#include "prgdev.h"
#include "gdat_tab.h"

#define PARSERControl \
    ((struct tParserControl *) ParserControl)

#define READCharacter(c) \
    { \
        READChar1532DataFile(c, PARSERControl->mDataFile, vEof); \
        PARSERControl->mFilePos++; \
        if (c == '\n') \
        { \
            PARSERControl->mLine++; \
            STRDone(PARSERControl->mLineBuf); \
            STRInit(PARSERControl->mLineBuf, ""); \
        } \
        else \
            STRAddChar(PARSERControl->mLineBuf, c); \
    }

#define UNREADCharacter(c) \
    { \
        UNREADChar1532DataFile(c, PARSERControl->mDataFile); \
        STRRemoveLastChar(PARSERControl->mLineBuf); \
        PARSERControl->mFilePos--; \
        if (c == '\n') \
        { \
            PARSERControl->mLine--; \
        } \
    }
#define FREEParserControl \
        { \
                STRDone(vParserControl.mStrBuf); \
                STRDone(vParserControl.mLineBuf); \
        }

#define STRToStack \
        { \
                STRInit(vStrBuf, STRPChar(PARSERControl->mStrBuf)); \
                AddListItem(&vStringBufferData, vStrBuf.mString); \
                lvalp->mTString = vStrBuf; \
        }

tList vStringBufferData;
struct tParserControl
{
    int mReadDate;                              // Enables token DATE
    int mReadHex;                                   // Enables token HEX
    int mReadString;                        // Enables token VHDL_STRING
    int mQuotes;                                // Indicates open string
    int mLine;                                      // Common line in parsing
    int mError;                                     // Indicates semantic error
    int mEmpty;                                     // Error if data file is empty
    TSTRING mStrBuf;              // String buffer
    TSTRING mLineBuf;                           // String buffer (common line)
    TFILE1532Data mDataFile;      // Data file
    pDevEnt mDevice;                            // Device
    long mFilePos;                      // Current file position
};

/* -------------------------------------------------------------------------- */
/* --- inits parser control and starts file parsing                       --- */
/* -------------------------------------------------------------------------- */
int parseDataFile(pDevEnt aDevice)
{
    struct tParserControl vParserControl;
    char *vFileName;
    int vError = rtOk;

    // init
    vParserControl.mReadDate = 0;
    vParserControl.mReadHex = 0;
    vParserControl.mReadString = 0;
    vParserControl.mQuotes = 0;
    vParserControl.mLine = 1;
    vParserControl.mError = rtOk;
    vParserControl.mEmpty = 0;
    STRInit(vParserControl.mStrBuf, "");
    STRInit(vParserControl.mLineBuf, "");
    vParserControl.mDevice = aDevice;
    vParserControl.mFilePos = 0;

  if (STRLen(DATAInputFileNameOfDevice(vParserControl.mDevice)) == 0)
  {
        FREEParserControl;
        return rtOk;
  }

    vFileName = STRPChar(DATAInputFileNameOfDevice(vParserControl.mDevice));
    if (vFileName == NULL)
        return rtOk;

    // open data file
    OPEN1532DataFileRO(vParserControl.mDataFile, vFileName);
    if (vParserControl.mDataFile == NULL)
    {
        WRITEError("...File \"%s\" not found (device %s).\n", vFileName,
            STRPChar(DEVICEName(vParserControl.mDevice)));
        FREEParserControl;
        return rtErr;
    }
    // parse (save file positions of data blocks)
    WRITEStatus("...Start parsing data file \"%s\" (device %s).\n", vFileName,
        STRPChar(DEVICEName(vParserControl.mDevice)));

    InitList(&vStringBufferData, loptFREE);
    SetListFunc(&vStringBufferData,NULL,NULL,(pfFreeItem)free );
    if (DATAparse((void *) &vParserControl))
    {
        WRITEError("\n    ...line:%d>%s\n", vParserControl.mLine,
            STRPChar(vParserControl.mLineBuf));
        vError = rtErr;
    }
    CLOSE1532DataFile(vParserControl.mDataFile);
    DoneList(&vStringBufferData);
    FREEParserControl;
    //STRDone(vParserControl.mStrBuf);
    //STRDone(vParserControl.mLineBuf);

    // return error if unsuccessfully parsing
    vParserControl.mError |= vError;
    if (vParserControl.mEmpty)
        vParserControl.mError |= rtErr;
    return vParserControl.mError;
}

/* -------------------------------------------------------------------------- */
/* --- parse error handling                                               --- */
/* -------------------------------------------------------------------------- */
int DATAerror(aErrorString)
{
    WRITEError("  ...parse error:");
    return 0;
}

/* -------------------------------------------------------------------------- */
/* --- scanner - get the tokens for parser                                --- */
/* -------------------------------------------------------------------------- */
int DATAlex(lvalp, ParserControl)
    YYSTYPE *lvalp;
{
    char c, ci;
    int vEof = 0;
    int i, vStrLen;
    int vIsHex, vIsAlpha;
    TSTRING vStrBuf;

    STRDone(PARSERControl->mStrBuf);

    // Ignore whitespace, get first nonwhite character.
    READCharacter(c);
    while (c == ' ' || c == '\t' || c == '\n')
        READCharacter(c);

    // Ignore Comments
    if (c == '-')
        c = OverreadComment(ParserControl);

    // Overread strings (-> creator)
    if (PARSERControl->mReadString)
    {
        do
        {
            // Add character to the buffer.
            STRAddChar(PARSERControl->mStrBuf, c);
            // Get another character.
            READCharacter(c);
            // Concat strings (Overread -"&"-)
            if ((c == '"') && PARSERControl->mQuotes)
                c = OverreadSringConcat(PARSERControl);
        }
        while ((c != '"') && (! vEof));

        UNREADCharacter(c);
        //STRInit((lvalp->mTString), STRPChar(PARSERControl->mStrBuf));
        STRToStack;
        return VHDL_STRING;
    }
    if (c == '"')
        PARSERControl->mQuotes = ! PARSERControl->mQuotes;

    // End of file
    if (vEof)
        return 0;

    // Read the string.
    if (CHARIsAlnum(c))
    {
        do
        {
            // Add this character to the buffer.
            STRAddChar(PARSERControl->mStrBuf, c);
            // Get another character.
            READCharacter(c);
        }
        while (! vEof && (CHARIsAlnum (c) || (c == '_')));

        UNREADCharacter(c);
        //STRInit((lvalp->mTString), STRPChar(PARSERControl->mStrBuf));
        STRToStack;

        // get the token
        // reserved words
        if (STRChEqualNoCase(PARSERControl->mStrBuf, "IEEE_1532_DATA"))
            return IEEE_1532_DATA;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "header"))
            return HEADER;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "std_version"))
            return STD_VERSION;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "STD_1532_2000"))
            return STD_1532_2000;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "entity"))
            return ENTITY;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "creation_date"))
            return CREATION_DATE;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "creator"))
            return CREATOR;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "data"))
            return DATA;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "Initialize"))
            return INITIALIZE;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "Repeat"))
            return REPEAT;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "Terminate"))
            return TERMINATE;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "data_crc"))
            return DATA_CRC;

/* Added "OVERRIDE" construct to data file
Frank Phillips @Xilinx on 03-07-01 */

       else if (STRChEqualNoCase(PARSERControl->mStrBuf, "Override"))
           return OVERRIDE;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "On"))
            return ON;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "Off"))
            return OFF;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "optional"))
            return OPTIONAL;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "recommended"))
            return RECOMMENDED;
        else if (STRChEqualNoCase(PARSERControl->mStrBuf, "unprocessed"))
            return UNPROCESSED;
 
        // date, hex or id ?
        vIsHex = 1;
        vIsAlpha = 0;
        vStrLen = STRLen(PARSERControl->mStrBuf);

    for (i = 0; i < vStrLen; i++)
        {
            ci = STRPChar(PARSERControl->mStrBuf)[i];
            if ((ci < '0') || ((ci > '9') && (ci < 'A')) || ((ci > 'F')
                    && (ci < 'a')) || (ci > 'f') || (c == '_'))
                vIsHex = 0;
            if (((ci >= 'A') && (ci <= 'Z')) || ((ci >= 'a') && (ci <= 'z'))
                    || (c == '-'))
                vIsAlpha = 1;
        }

        // DATE
        if ((c == '.') && (! vIsAlpha) && PARSERControl->mReadDate)
        {
            // yyyymmdd.
            if (STRLen(PARSERControl->mStrBuf) == 8)
            {
                // .
                READCharacter(c);
                STRAddChar(PARSERControl->mStrBuf, c);
                // .hhmm
                READCharacter(c);
                while (CHARIsDigit(c))
                {
                    STRAddChar(PARSERControl->mStrBuf, c);
                    READCharacter(c);
                }
                // yyyymmdd.hhmm
                if (STRLen(PARSERControl->mStrBuf) == 13 && (! CHARIsAlnum(c))
                        && (c != '_'))
                {
                    READCharacter(c);
                    //STRInit((lvalp->mTString), STRPChar(PARSERControl->mStrBuf));
                    STRToStack;
                    return DATE;
                }
            }
            // no date
            return INVALID;
        }

        // HEX
        if (PARSERControl->mReadHex && vIsHex)
            return HEX;

        // VHDL_ID
        if (CHARIsAlpha(STRPChar(PARSERControl->mStrBuf)[0]))
            return VHDL_ID;

        // Invalid string
        if (vStrLen > 1)
            return INVALID;
    }
    /* Any other character is a token by itself.        */
    return c;
}

/* -------------------------------------------------------------------------- */
/* --- ignores comments                                                   --- */
/* -------------------------------------------------------------------------- */
static int OverreadComment(ParserControl)
{
    char c = '-';
    int vEof;

    while (c == '-')
    {
        // Overread multiple comments
        READCharacter(c);
        if (c == '-')
        {
            // Overread comment
            while (c != '\n')
                READCharacter(c);
            while (c == ' ' || c == '\t' || c == '\n')
                READCharacter(c);
        }
        else
        {
            // No comment -> return token '-'
            UNREADCharacter(c);
            return '-';
        }
    }
    return c;
}

/* -------------------------------------------------------------------------- */
/* --- concat bsdl strings (ignores -"&"-)                                --- */
/* -------------------------------------------------------------------------- */
static int OverreadSringConcat(ParserControl)
{
    char c;
    int vEof;

    // Ignore whitespace, get first nonwhite character.
    READCharacter(c);
    while (c == ' ' || c == '\t' || c == '\n')
        READCharacter(c);

    // Ignore Comments
    if (c == '-')
        c = OverreadComment(ParserControl);
    while (c == ' ' || c == '\t' || c == '\n')
        READCharacter(c);

    // No string concatenation
    if (c != '&')
    {
        UNREADCharacter(c);
        return '"';
    }

    // Ignore whitespace, get first nonwhite character.
    READCharacter(c);
    while (c == ' ' || c == '\t' || c == '\n')
        READCharacter(c);

    // Ignore Comments
    if (c == '-')
        c = OverreadComment(ParserControl);
    while (c == ' ' || c == '\t' || c == '\n')
        READCharacter(c);

    // No string concatenation
    if (c != '"')
    {
        UNREADCharacter(c);
        UNREADCharacter('&');
        return '"';
    }

    // String concatenation overread
    READCharacter(c);
    return c;
}

/***** Read single hex strings ************************************************/

#define READCharacter(c) \
    { \
        READChar1532DataFile(c, apStatus->mDataInputFile, vEof); \
    }

#define UNREADCharacter(c) \
    { \
        UNREADChar1532DataFile(c, apStatus->mDataInputFile); \
    }

/* -------------------------------------------------------------------------- */
/* --- reads single hex string from data file                             --- */
/* -------------------------------------------------------------------------- */
int GetNextInputHexString(pActionStatus apStatus)
{
    char c, ci;
    int i, vStrLen;
    int vIsHex = 1;
    int vEof;

  long vCharCount = 0;
    long vLineCount = 0;

    STRDone(apStatus->mInputHexString);
    apStatus->mBlockEnd = 0;

    // get first hex character.
    READCharacter(c);
    while ((! CHARIsAlnum(c)) && (! vEof))
    {
        if (c == ')')
        {
            WRITEError("\nError - not enough input data for a full loop iteration.\n");
            return rtErr;
        }
        READCharacter(c);
    }

    // abort if end of file
    if (vEof)
    {
        WRITEError("\nError - EOF while reading next hex string.\n");
        return rtErr;
    }

    // read the string
    if (CHARIsAlnum(c))
    {
        do
        {
            // Add this character to the buffer.
            if (CHARIsAlnum(c))
                STRAddChar(apStatus->mInputHexString, c);
            // Get another character.
            READCharacter(c);
            if (vEof)
                return rtErr;
        }
        while (CHARIsAlnum(c) || (c == ' ') || (c == '\t') || (c == '\n'));

        // is it a hex string ?
        vStrLen = STRLen(apStatus->mInputHexString);
        for (i = 0; i < vStrLen; i++)
        {
            ci = STRPChar(apStatus->mInputHexString)[i];
            if ((ci < '0') || ((ci > '9') && (ci < 'A')) || ((ci > 'F')
                    && (ci < 'a')) || (ci > 'f'))
                vIsHex = 0;
        }
    }
  // check block end
    while ((! CHARIsAlnum(c)) && (! vEof) && (c != ')'))
         READCharacter(c);
    if (c == ')')
            apStatus->mBlockEnd = 1;
    UNREADCharacter(c);

    // no hex string
    if (! vIsHex)
    {
        WRITEError("\nError - reading invalid hex string (%s) from data file.\n",
            STRPChar(apStatus->mInputHexString));
        STRInit(apStatus->mInputHexString, "");
        return rtErr;
    }
    return rtOk;
}
