/*****************************************************************************
/* J Drive (tm) Engine
/* Copyright 2000, Xilinx, Inc.
/* The use, reproduction, and distribution of this code is subject
/* to the J Drive license requirements and restrictions.
*****************************************************************************/
// Interface to one pdevice
#include "libenv.h"
#include "prgtyp.h"
#include "prgdev.h"
#include "freedev.h"

/***** init data **************************************************************/
struct IscInstStruc ISCInstrList[] =
{
    { ISCMand, "ISC_ENABLE"          },
    { ISCMand, "ISC_DISABLE"         },
    { ISCMand, "ISC_PROGRAM"         },
    { ISCMand, "ISC_NOOP"            },
    { ISCOpt,  "ISC_READ"            },
    { ISCOpt,  "ISC_ERASE"           },
    { ISCOpt,  "ISC_PROGRAM_USERCODE"},
    { ISCOpt,  "ISC_DISCHARGE"       },
    { ISCOpt,  "ISC_PROGRAM_DONE"    },
    { ISCOpt,  "ISC_ERASE_DONE"      },
    { ISCOpt,  "ISC_PROGRAM_SECURITY"},
    { ISCOpt,  "ISC_READ_INFO"       },
    { ISCOpt,  "ISC_DATA_SHIFT"      },
    { ISCOpt,  "ISC_SETUP"           },
    { ISCOpt,  "ISC_ADDRESS_SHIFT"   },
    { ISCOpt,  "ISC_INCREMENT"       }
};

/******************************************************************************/
/***** init and free                                                      *****/
/******************************************************************************/

/* -------------------------------------------------------------------------- */
pDevEnt InitDevice(PSTRING aName)
{
	pDevEnt pDev;      // Pointer to one Device
	pfCompare vfComp;  // Comparefunction for List
	pfKeyOf   vfKeyOf; // KeyOf Function  for List
   pfFreeItem vfFreeItem;
	int vSize = sizeof(tDevEnt);

	pDev = (pDevEnt)GETMem(vSize);
	if (pDev != NULL)
	{
		FILLChar(pDev,0, vSize);

		// BSDL-Info
		if (pDev->dveBsdlInfo == NULL)
		{
			pDev->dveBsdlInfo = InitBsdlInfo();
			// list of instructions
			vfKeyOf = KeyOfInstList;
			vfComp  = ComparepString;
         vfFreeItem = FreeInstruction;
			SetListFunc(GDvInstList(pDev),vfComp,vfKeyOf,vfFreeItem);
			// list of data registers
			vfKeyOf = KeyOfDRList;
         vfFreeItem = FreeDataReg;
			SetListFunc(GDvDRegList(pDev),vfComp,vfKeyOf,vfFreeItem);
			// list of procedures
			vfKeyOf = KeyOfProcList;
			vfComp  = CompareProc;
         vfFreeItem = FreeProc;
			SetListFunc(GDvProcList(pDev),vfComp,vfKeyOf,vfFreeItem);
			// list of flows
			vfKeyOf = KeyOfFlowList;
			vfComp  = CompareIscFlow;
          vfFreeItem = FreeFlow;
			SetListFunc(GDvIscFlowList(pDev),vfComp,vfKeyOf,vfFreeItem);
			// list of actions
			vfKeyOf = KeyOfAction;
			vfComp = CompareAction;
          vfFreeItem = FreeAction;
			SetListFunc(GDvActionList(pDev), vfComp,vfKeyOf,vfFreeItem);
		}

		STRInit(DEVICEName(pDev), "");
		STRInit(BSDLFileNameOfDevice(pDev), "");
		STRInit(DATAInputFileNameOfDevice(pDev), "");
		STRInit(DATAOutputFileNameOfDevice(pDev), "");
		STRInit(ACTIONNameOfDevice(pDev), "");
		STRInit(DATANameOfDevice(pDev), "");
		STRInit(pDev->mSemantikErrors, "");

		// action status
		pDev->mActionStatus = InitActionStatus();
		// . list of data records
		vfKeyOf = KeyOfDataRecordList;
		vfComp  = ComparepString;
       vfFreeItem = FreeDataRecord;
		SetListFunc(GDvDataRecordList(pDev),vfComp,vfKeyOf,vfFreeItem);
		// device name
		STRAssign(pDev->dveName, *aName);
	}
	return(pDev);
}

/* -------------------------------------------------------------------------- */
void FreeDevice(void  *apDevice)
{  pDevEnt  vDevice = apDevice;
	if (apDevice != NULL)
	{
		STRDone(DEVICEName(vDevice));
		STRDone(BSDLFileNameOfDevice(vDevice));
		STRDone(DATAInputFileNameOfDevice(vDevice));
		STRDone(DATAOutputFileNameOfDevice(vDevice));
		STRDone(ACTIONNameOfDevice(vDevice));
		STRDone(DATANameOfDevice(vDevice));
		STRDone(vDevice->mSemantikErrors);

		FreeBsdlInfo(vDevice->dveBsdlInfo);
		FreeActionStatus(vDevice->mActionStatus);
		FREEMem(apDevice, sizeof(tDevEnt));
	}
}

/* -------------------------------------------------------------------------- */
pBsdlInfo InitBsdlInfo(void)
{
	pBsdlInfo pInf;
	pInf = (pBsdlInfo) GETMem(sizeof(tBsdlInfo));
	FILLChar(pInf,0,sizeof(tBsdlInfo));

	if (pInf!=NULL)
	{
		InitList(&pInf->bsdDataRegList,loptFREE);
		InitList(&pInf->bsdInstList,loptFREE);  //Instructions of a device
		InitList(&pInf->bsdActionList,loptFREE);
		InitList(&pInf->bsdProcList,loptFREE);
		InitList(&pInf->bsdIscFlowList,loptFREE);

		STRInit(pInf->UserCode, "");
		STRInit(pInf->IdCode, "");
		STRInit(pInf->dveIrCap, "");
		STRInit(pInf->BlankUserCode, "");
	}
	return(pInf);
}

/* -------------------------------------------------------------------------- */
void FreeBsdlInfo(pBsdlInfo apBsdlInfo)
{
	if (apBsdlInfo != NULL)
	{
		DoneList(&apBsdlInfo->bsdDataRegList);
		DoneList(&apBsdlInfo->bsdInstList);
		DoneList(&apBsdlInfo->bsdActionList);
		DoneList(&apBsdlInfo->bsdProcList);
		DoneList(&apBsdlInfo->bsdIscFlowList);

		STRDone(apBsdlInfo->UserCode);
		STRDone(apBsdlInfo->IdCode);
		STRDone(apBsdlInfo->dveIrCap);
		STRDone(apBsdlInfo->BlankUserCode);

		FREEMem(apBsdlInfo, sizeof(tBsdlInfo));
	}
}

/* -------------------------------------------------------------------------- */
pActionStatus InitActionStatus(void)
{
	pActionStatus vpActionStatus = (pActionStatus)GETMem(sizeof(tActionStatus));
	if (vpActionStatus != NULL)
		FILLChar(vpActionStatus, 0, sizeof(tActionStatus));

	InitList(&vpActionStatus->mDataRecordList,loptFREE);
/* Frank Phillips @ Xilinx Added the following two lines on
May_01_2001 */

        STRInit(vpActionStatus->mEnableInstruction, "");
        STRInit(vpActionStatus->mDisableInstruction, "");

	STRInit(vpActionStatus->mBypassInstruction, "");
	STRInit(vpActionStatus->mInstruction, "");
	STRInit(vpActionStatus->mInputRecordName, "");
	STRInit(vpActionStatus->mInputHexString, "");
	#ifdef DEBUG
		STRInit(vpActionStatus->mUFValueType, "");
		STRInit(vpActionStatus->mCFValueType, "");
		STRInit(vpActionStatus->mMaskValueType, "");
	#endif
	return vpActionStatus;
}

/* -------------------------------------------------------------------------- */
void FreeActionStatus(pActionStatus apActionStatus)
{
	DoneList(&apActionStatus->mDataRecordList);

	STRDone(apActionStatus->mBypassInstruction);
//	STRDone(apActionStatus->mInstruction);
	STRDone(apActionStatus->mInputRecordName);
	STRDone(apActionStatus->mInputHexString);
	#ifdef DEBUG
		STRDone(apActionStatus->mUFValueType);
		STRDone(apActionStatus->mCFValueType);
		STRDone(apActionStatus->mMaskValueType);
	#endif
	FREEMem(apActionStatus, sizeof(tActionStatus));
}

/* -------------------------------------------------------------------------- */
pDataRegEnt InitDataReg(void)
{
	pDataRegEnt vpDataRegEnt = (pDataRegEnt)GETMem(sizeof(tDataRegEnt));
	if (vpDataRegEnt != NULL)
		FILLChar(vpDataRegEnt, 0, sizeof(tDataRegEnt));

	STRInit(vpDataRegEnt->mDataRegName, "");
	return vpDataRegEnt;
}

/* -------------------------------------------------------------------------- */

/* -------------------------------------------------------------------------- */
pActionEnt InitAction(PSTRING aName,PSTRING aDataName)
{
	//TSTRING vName = *aName;
	//TSTRING vDataName = *aDataName;
	pActionEnt vAct = (pActionEnt)GETMem(sizeof(tActionEnt));
	if (vAct != NULL)
	{
	//	STRMove(vAct->aceName, vName);
 //		STRMove(vAct->aceDataName, vDataName);
		STRInit(vAct->aceName, STRPChar(*aName));
		STRInit(vAct->aceDataName, STRPChar(*aDataName));
		InitList(&vAct->aceProcList,loptFREE);
		SetListFunc(&vAct->aceProcList, NULL, NULL, (pfFreeItem)FreeActProc);

 }
	return(vAct);
}

/* -------------------------------------------------------------------------- */
void FreeAction(void *apActionEnt)
{  pActionEnt vActionEnt = apActionEnt;
	STRDone(vActionEnt->aceName);
	STRDone(vActionEnt->aceDataName);

	DoneList(&vActionEnt->aceProcList);
	FREEMem(apActionEnt, sizeof(tActionEnt));
   apActionEnt = NULL;
}

/* -------------------------------------------------------------------------- */
pProcEnt InitProc(PSTRING aName, PSTRING aDataName,int aLine)
{
	TSTRING vName = *aName;
	TSTRING vDataName = *aDataName;
	pProcEnt vProc = (pProcEnt)GETMem(sizeof(tProcEnt));

	if (vProc != NULL) {
		vProc->preLine = aLine;
		FILLChar(vProc, 0, sizeof(tProcEnt));
		STRMove(vProc->preName,vName);
		STRMove(vProc->preDataName,vDataName);
		InitList(&vProc->preFlowList, 0);
	};
 return(vProc);
}

/* -------------------------------------------------------------------------- */
void FreeProc (void *aProc)
{
	if (aProc != NULL)
	{
		STRDone(((pProcEnt)aProc)->preName);
		STRDone(((pProcEnt)aProc)->preDataName);
		DoneList(&((pProcEnt)aProc)->preFlowList);

		FREEMem(aProc,sizeof(tProcEnt));
	}
}

/* -------------------------------------------------------------------------- */
pActProcEnt InitActProc(PSTRING aName,PSTRING aDataName)
{ 
    TSTRING vName = *aName;
	TSTRING vDataName = *aDataName;
    pActProcEnt vActProc = (pActProcEnt)GETMem(sizeof(tActProcEnt));
	if (vActProc != NULL) {
		FILLChar(vActProc,0, sizeof(tActProcEnt));
		STRMove(vActProc->aseProcName,vName);
		STRMove(vActProc->aseDataName,vDataName);
	}
	return(vActProc);
 };

/* -------------------------------------------------------------------------- */
FreeActProc(pActProcEnt aActProc)
{
	if (aActProc != NULL) {
		STRDone(aActProc->aseProcName);
		STRDone(aActProc->aseDataName);
		//!! not free the pointer   aseProt !!
		FREEMem(aActProc, sizeof(tActProcEnt));
	};
};

/* -------------------------------------------------------------------------- */
pIscFlowEnt InitFlow(PSTRING aFlowName, PSTRING aDataName, int aLine)
{ pfCompare vfComp;  // Comparefunction for List
  pfKeyOf   vfKeyOf; // KeyOf Function  for List
  pfFreeItem vfFreeItem;
	TSTRING Str;
	pIscFlowEnt vFlow =NULL;
	vFlow = (pIscFlowEnt)GETMem(sizeof(tIscFlowEnt));
	FILLChar(vFlow,0,sizeof(tIscFlowEnt));
	vFlow->mLine = aLine;
	Str = *aFlowName;

 STRMove(vFlow->ifeFlowName,Str);
 Str = *aDataName;
 STRMove(vFlow->ifeDataName,Str);
 InitList(&vFlow->VarStack,loptFREE);
 vfKeyOf = KeyOfVarStack;
 vfComp  = ComparepString;
 vfFreeItem = FreeVar;
 SetListFunc(&vFlow->VarStack,vfComp,vfKeyOf,vfFreeItem);
 return(vFlow);
};

/* -------------------------------------------------------------------------- */
void FreeFlow(void *aFlow)
{  
    pIscFlowEnt vFlow = (pIscFlowEnt)aFlow;
	if (vFlow) {
	  DoneList(&vFlow->VarStack);
	  if (vFlow->ifeBlocks[bkInit])
	     FreeExecBlock(vFlow->ifeBlocks[bkInit]);
	  if (vFlow->ifeBlocks[bkRepeat])
	     FreeExecBlock(vFlow->ifeBlocks[bkRepeat]);
	  if (vFlow->ifeBlocks[bkTerminate])
	     FreeExecBlock(vFlow->ifeBlocks[bkTerminate]);
      STRDone(vFlow->ifeFlowName);
	  STRDone(vFlow->ifeDataName);
	}
	FREEMem(aFlow,sizeof(tIscFlowEnt));
   aFlow = NULL;
}

/* -------------------------------------------------------------------------- */
pExecBlock InitExecBlock(PSTRING aName,int aLoopCount)
{pExecBlock vBlock;
//	TSTRING Str = *aName;
	vBlock = (pExecBlock)GETMem(sizeof(tExecBlock));
	FILLChar(vBlock, 0, sizeof(tExecBlock));
	if (vBlock!=NULL) {
		vBlock->exbName = *aName;
	vBlock->exbLoopCount = aLoopCount;
	InitList(&vBlock->exbActList,loptFREE);
/* vfKeyOf = KeyOfVarStack;
 vfComp  = ComparepString;
 vfFreeItem = FreeFlowAct;
 SetListFunc(&vBlock->exbActList,vfComp,vfKeyOf,vfFreeItem);
*/
    SetListFunc(&vBlock->exbActList, NULL, NULL, (pfFreeItem)FreeFlowAct);
	}
	return(vBlock);
};

/* -------------------------------------------------------------------------- */
void FreeExecBlock(pExecBlock aBlock)
{
	if (aBlock !=NULL) 
	{
	  STRDone(aBlock->exbName);
	  DoneList(&aBlock->exbActList);
	  FREEMem(aBlock, sizeof(tExecBlock));
	}
};

/* -------------------------------------------------------------------------- */
pFlowActEnt InitFlowAct(PSTRING aName)
{TSTRING Str = *aName;
 pFlowActEnt vAct = (pFlowActEnt)GETMem(sizeof(tFlowActEnt));
 FILLChar(vAct,0,sizeof(tFlowActEnt));
 if (vAct!=NULL)  {
	 STRMove(vAct->facInstName,Str);
    InitList(&vAct->UpdateFieldList,loptFREE);
    SetListFunc(&vAct->UpdateFieldList, NULL, NULL, (pfFreeItem)FreeUpField);
    InitList(&vAct->CaptureFieldList,loptFREE);
    SetListFunc(&vAct->CaptureFieldList, NULL, NULL, (pfFreeItem)FreeCapField);
 };
 return(vAct);
};

/* -------------------------------------------------------------------------- */
void FreeFlowAct(pFlowActEnt aAct)
{
 if (aAct!=NULL)  {
     STRDone(aAct->facInstName);
	 // Free Wait Spec
	 STRDone(aAct->facWaitMin.mPortId);
     STRDone(aAct->facWaitMax.mPortId);
 	 DoneList(&aAct->UpdateFieldList);
	 DoneList(&aAct->CaptureFieldList);
     FREEMem(aAct, sizeof(tFlowActEnt));
 };
};

/* -------------------------------------------------------------------------- */
pUpFieldEnt InitUpField(int aWidth, pDataExpr aExpr)
{
   pUpFieldEnt vUpField = (pUpFieldEnt)GETMem(sizeof(tUpFieldEnt));

   if (vUpField != NULL) {
     vUpField->Width = aWidth;
	 vUpField->DataExpr = *aExpr;
  //       InitBitField(&(vUpField->mValue),0); 
	 vUpField->mValue.mCount = 0;
	 vUpField->mValue.mSize  =  0;
	 vUpField->mValue.mBytes = NULL;

	 STRInit(vUpField->DataExpr.mHexStringOperand, "");
	 if ((aExpr->mOpType == opHexStringConstant) || (aExpr->mOpType == opAssignment))
	    STRAssign(vUpField->DataExpr.mHexStringOperand, aExpr->mHexStringOperand);
	 if (aExpr->mOpType == opHexStringConstant)
        HexToBitField(&(vUpField->mValue), &(aExpr->mHexStringOperand), aWidth);		
   }
   return(vUpField);
};

/* -------------------------------------------------------------------------- */
FreeUpField(pUpFieldEnt aUpField)
{
  if (aUpField != NULL)
  {
    switch (aUpField->DataExpr.mOpType)
	{
      case opInput: if (aUpField->DataExpr.mVarEnt)
	                      break;  // else DoneBitField 
      case opHexStringConstant:
	                DoneBitField(&(aUpField->mValue));
	}
	STRDone(aUpField->DataExpr.mHexStringOperand);
    FREEMem(aUpField,sizeof(tUpFieldEnt));
  }
};

/* -------------------------------------------------------------------------- */
pCapFieldEnt InitCapField(int aWidth, pDataExpr aExpr)
{
  pCapFieldEnt vCapField = (pCapFieldEnt)GETMem(sizeof(tCapFieldEnt));

  if (vCapField != NULL) {
    vCapField->Width = aWidth;
    vCapField->ExpectData = *aExpr;
//        InitBitField(&(vCapField->mValue),0); 
	 vCapField->mValue.mCount = 0;
	 vCapField->mValue.mSize  =  0;
	 vCapField->mValue.mBytes = NULL;

	STRInit(vCapField->ExpectData.mHexStringOperand, "");
	STRInit(vCapField->CompMask.mHexStringOperand, "");
	if ((aExpr->mOpType == opHexStringConstant) || (aExpr->mOpType == opAssignment))
	    STRAssign(vCapField->ExpectData.mHexStringOperand, aExpr->mHexStringOperand);

    if (aExpr->mOpType == opHexStringConstant)
        HexToBitField(&(vCapField->mValue), &(aExpr->mHexStringOperand), aWidth);		

    vCapField->CompMask.mOpType = opHexStringConstant;
	InitBitField(&(vCapField->mMaskValue), aWidth);
	ComplementOfBitField(&(vCapField->mMaskValue));

  };
  return(vCapField);
};

/* -------------------------------------------------------------------------- */
FreeCapField( pCapFieldEnt aCapField)
{
  if (aCapField !=NULL)
  {

    switch (aCapField->ExpectData.mOpType)
    {
       case opInput: if (aCapField->ExpectData.mVarEnt)
	                      break;  // else DoneBitField 
       case opHexStringConstant:
	       DoneBitField(&(aCapField->mValue));
	}
	STRDone(aCapField->ExpectData.mHexStringOperand);

	switch (aCapField->CompMask.mOpType)
	{
       case opInput: if (aCapField->CompMask.mVarEnt)
	                      break;  // else DoneBitField 
	   case opHexStringConstant:
	       DoneBitField(&(aCapField->mMaskValue));
	}
	STRDone(aCapField->CompMask.mHexStringOperand);

    FREEMem(aCapField,sizeof(tCapFieldEnt));
   }
};

/* -------------------------------------------------------------------------- */
pDataExpr InitDataExpr(void)
{pDataExpr vDataExpr = (pDataExpr)GETMem(sizeof(tDataExpr));

	if (vDataExpr != NULL)
		FILLChar(vDataExpr, 0, sizeof(tDataExpr));
	return vDataExpr;
}

/* -------------------------------------------------------------------------- */
void DoneDataExpr(pDataExpr aDataExpr)
{
	if (aDataExpr != NULL)
	{
   	    STRDone(aDataExpr->mHexStringOperand);
		STRDone(aDataExpr->mVariable);
		FILLChar(aDataExpr, 0, sizeof(tDataExpr));
	}
}
/* -------------------------------------------------------------------------- */
void FreeDataExpr(pDataExpr aDataExpr)
{
	if (aDataExpr != NULL)
	{
		DoneDataExpr(aDataExpr);
		FREEMem(aDataExpr, sizeof(tDataExpr));
	}
}

/* -------------------------------------------------------------------------- */
pDurationSpec InitDurationSpec(void)
{pDurationSpec vDurationSpec = (pDurationSpec)GETMem(sizeof(tDurationSpec));
	if (vDurationSpec !=NULL)
		FILLChar(vDurationSpec, 0, sizeof(tDurationSpec));
	return vDurationSpec;
}

/* -------------------------------------------------------------------------- */
void DoneDurationSpec(pDurationSpec aDurationSpec)
{
	if (aDurationSpec != NULL)
		FILLChar(aDurationSpec, 0, sizeof(tDurationSpec));
}

/* -------------------------------------------------------------------------- */
void FreeDurationSpec(pDurationSpec aDurationSpec)
{
	if (aDurationSpec != NULL)
		FREEMem(aDurationSpec, sizeof(tDurationSpec));
}

/* -------------------------------------------------------------------------- */
pVarEnt InitVar(PSTRING aName, int aSize)
{TSTRING vName = *aName;

 pVarEnt vVar = (pVarEnt)GETMem(sizeof(tVarEnt));
 if (vVar != NULL)  {
	 FILLChar(vVar,0,sizeof(tVarEnt));
	 STRMove(vVar->varName,vName);
	 vVar->varSize = aSize;
 };
 return(vVar);
};

/* -------------------------------------------------------------------------- */
void FreeVar(void  *aVar)

{
	if (aVar !=NULL)
	{
		STRDone( ((pVarEnt)aVar)->varName);
		DoneBitField(&((pVarEnt)aVar)->varValue);
		FREEMem(aVar,sizeof(tVarEnt));
	}
};

/* -------------------------------------------------------------------------- */
pDataRecord InitDataRecord(PSTRING aDataName)
{
	pDataRecord vpDataRecord = (pDataRecord)GETMem(sizeof(tDataRecord));

    STRInit(vpDataRecord->mDataName, "");
	STRAssign(vpDataRecord->mDataName, *aDataName);
	vpDataRecord->mInitBlockPos = -1;
	vpDataRecord->mRepeatBlockPos = -1;
	vpDataRecord->mTerminateBlockPos = -1;
	return vpDataRecord;
};
/* -------------------------------------------------------------------------- */
void DoneDataRecord(pDataRecord apDataRecord)
{
	if (apDataRecord != NULL)
	{
	    STRDone(apDataRecord->mDataName);
		FILLChar(apDataRecord, 0, sizeof(tDataRecord));
	}
}
/* -------------------------------------------------------------------------- */
/*void FreeDataRecord(pDataRecord apDataRecord)
{
	DoneDataRecord(apDataRecord);
	if (apDataRecord != NULL)
	   FREEMem(apDataRecord, sizeof(tDataRecord));
}	*/


/***** list handling **********************************************************/

/* -------------------------------------------------------------------------- */
// Insert a new variable to the variable stack
int AddVar(pList apList, PSTRING aName, int aSize, int *aIndex)
{
	pVarEnt  vpVar;
	int vIdx = cnoId;
	int rt = rtOk;

	if (apList != NULL)
	{
		if (! lSearchList(apList, aName, &vIdx))
		{
			vpVar = InitVar(aName, aSize);
			if (vpVar != NULL)
				AddListItem(apList, vpVar);
			*aIndex = ITEMCountOfList(apList) - 1;
		}
		else
			*aIndex = vIdx;
	}
	else
	{
		rt = rtErr;
		WRITEError("Variable Stack Error \n");
	}
	return(rt);
}

/* -------------------------------------------------------------------------- */
pList GetCurrentActList(pDevEnt aDevEnt)
{pList vFlowList = GDvIscFlowList(aDevEnt);
 int vCount = vFlowList->Count;
 int vBlockId = cnoId;
 pIscFlowEnt vFlow	 = (pIscFlowEnt)vFlowList->items[vCount-1];
 if (vFlow !=NULL)  {
   vBlockId = vFlow->currBlockId;
	 return(&vFlow->ifeBlocks[vBlockId]->exbActList);
 };

};

// **************************************************//
// Functions called from command file parser
// Files
/* -------------------------------------------------------------------------- */
void SetBsdlFileName(pDevEnt aDevEnt, PSTRING aFileName)
{
	TSTRING vStr = *aFileName;
	if (aDevEnt != NULL)
		STRMove(aDevEnt ->dveBsdlFileName, vStr);
};

/* -------------------------------------------------------------------------- */
void SetDataInputFileName(pDevEnt aDevEnt, PSTRING aFileName)
{
	TSTRING vStr = *aFileName;
	if (aDevEnt != NULL)
		STRMove(aDevEnt ->dveDFileInpName, vStr);
}

/* -------------------------------------------------------------------------- */
void SetDataOutputFileName(pDevEnt aDevEnt, PSTRING aFileName)
{
	TSTRING vStr = *aFileName;
	if (aDevEnt != NULL)
		STRMove(aDevEnt ->dveDFileOutName, vStr);
}

/* -------------------------------------------------------------------------- */
void SetActionName(pDevEnt aDevEnt, PSTRING aActionName)
{
	TSTRING vStr = *aActionName;
	if (aDevEnt != NULL)
		//STRMove(aDevEnt ->dveActionName, vStr);
		STRAssign(aDevEnt ->dveActionName, vStr);
}

/* -------------------------------------------------------------------------- */
void SetDataName(pDevEnt aDevEnt, PSTRING aDataName)
{
	TSTRING vStr = *aDataName;
	if (aDevEnt != NULL)
		//STRMove(aDevEnt ->dveDataName, vStr);
		STRAssign(aDevEnt ->dveDataName, vStr);
}

// **************************************************//
// Functions called from bsdl file parser
// Standard IEEE_1149.1_1994:

/* -------------------------------------------------------------------------- */
int SetTCKSignalStatus(pDevEnt apDevEnt, int aStatus, int aLine)
{
	if (apDevEnt != NULL)
		DEVICETCKSignalStatus(apDevEnt) = aStatus;
	return rtOk;
}

/* -------------------------------------------------------------------------- */
// Instruction-Register
int PutIrLength(pDevEnt apDevEnt, int aIrLength, int aLine)
{
	if (apDevEnt != NULL)
		IRLengthOfDevice(apDevEnt) = aIrLength;
	return rtOk;
}

/* -------------------------------------------------------------------------- */
int AddInstruction(pDevEnt aDevEnt, PSTRING aInst, int aLine)
{
	int rt = rtOk;
	pInstructionEnt p;
	TSTRING Str;

	if (aInst!= NULL)
	{
			p = (pInstructionEnt) GETMem(sizeof(tInstructionEnt));
	 if (p!=NULL) {
			Str = *aInst;
		STRMove(p->mInstructionName, Str);
		 };
		AddListItem(&((aDevEnt)->dveBsdlInfo->bsdInstList),(void *)p);
	 };
 return(rt);
};
void FreeInstruction(void *aInst) 
{
  if (aInst)
  {
    STRDone( ((pInstructionEnt)aInst) ->mInstructionName);
    STRDone( ((pInstructionEnt)aInst) ->mInstructionOpCode);
    FREEMem(aInst,sizeof(tInstructionEnt));
  }
};
/* -------------------------------------------------------------------------- */
int AddOpCode(pDevEnt aDevEnt,PSTRING aCode, int aLine)
{int rt = rtOk;
 TSTRING Str = *aCode;
 pList vInstList = GDvInstList(aDevEnt);
 pInstructionEnt vInst= NULL;
 int vCount = vInstList->Count;
	if (vCount > 0)
		vInst = (pInstructionEnt)vInstList->items[vCount -1];
	if (vInst !=NULL) {
    // Set only first Instruction
		STRMove(vInst->mInstructionOpCode,Str);
  };
	// Check Instruction length of String
	if ((aDevEnt)->dveIrLength != vInst->mInstructionOpCode.mLength)
	{
		STRChConcat(aDevEnt->mSemantikErrors, "    ...line ");
		STRIntConcat(aDevEnt->mSemantikErrors, aLine);
		STRChConcat(aDevEnt->mSemantikErrors, ": instruction length mismatch.\n");
		rt = rtErr;
	 };
	return(rt);
};

/* -------------------------------------------------------------------------- */
int AddIrCapVal(pDevEnt apDevice, PSTRING aIrCap, int aLine)
{
	TSTRING Str = *aIrCap;
	if (STRChEqualNoCase(DEVICECaptureIr(apDevice), ""))
		STRMove(DEVICECaptureIr(apDevice), Str);
	return rtOk;
}

/* -------------------------------------------------------------------------- */
// Id and user code register
int AddIdCode(pDevEnt apDevice, PSTRING aCode, int aLine)
{
	TSTRING Str = * aCode;
	if (STRChEqualNoCase(DEVICEIdCode(apDevice), ""))
		STRMove(DEVICEIdCode(apDevice), Str);
	return rtOk;
}

/* -------------------------------------------------------------------------- */
int AddUserCode(pDevEnt apDevice, PSTRING aCode, int aLine)
{
	TSTRING Str = * aCode;
	if (STRChEqualNoCase(DEVICEUserCode(apDevice), ""))
		STRMove(DEVICEUserCode(apDevice), Str);
	return rtErr;
}

/* -------------------------------------------------------------------------- */
// Register access
int AddDataReg(pDevEnt apDevEnt, PSTRING aName, int aLength, int aLine)
{
	pList vpDRList = GDvDRegList(apDevEnt);
	pDataRegEnt vpDR;
  int vCount = cnoId;
  TSTRING Str = *aName;

	if (apDevEnt != NULL)
	{
		if (vpDRList!=NULL)
			vCount = vpDRList->Count;
	//	vpDR = (pDataRegEnt)GETMem(sizeof(tDataRegEnt));
		vpDR = InitDataReg();
		vpDR->mId = vCount;
		STRMove(vpDR->mDataRegName,Str);
		vpDR->mLength = aLength;
		AddListItem(vpDRList,(void *)vpDR);
		return rtOk;
	}
	return rtErr;
}

/* -------------------------------------------------------------------------- */
// . aName = Instruction Name
int AddInstToDR(pDevEnt aDevEnt, PSTRING aName, PSTRING aCapVal, int aLine)
{int rt = rtOk;
 int vIdx = 0; // Search index
 pList InstList = GDvInstList(aDevEnt);
 pList DrList   = GDvDRegList(aDevEnt);
 pInstructionEnt  vInst;
 TSTRING vCap = *aCapVal;

	if (aDevEnt != NULL) {
			if (lSearchList(InstList, aName, &vIdx)) {
			vInst = (pInstructionEnt)InstList->items[vIdx];
			if (vInst!=NULL) {
				STRMove(vInst->mDataRegCapValue,vCap);
				vInst->mDataRegId = vIdx;
				vInst->mpDataReg  = (pDataRegEnt)InstList->items[vIdx];
			};
		}
		else
		{
			// error string
			STRChConcat(aDevEnt->mSemantikErrors, "    ...line ");
			STRIntConcat(aDevEnt->mSemantikErrors, aLine);
			STRChConcat(aDevEnt->mSemantikErrors, ": instruction ");
			STRConcat(aDevEnt->mSemantikErrors, *aName);
			STRChConcat(aDevEnt->mSemantikErrors, " not found.\n");
		};
	};
	return(rt);
};

// **************************************************//
// Functions called from bsdl file parser
// Standard IEEE_1532_2000:
// Isc status and pin behavior

/* -------------------------------------------------------------------------- */
int SetStatus(pDevEnt apDevice, int aStatus, int aLine)
{
	DEVICEIscStatus(apDevice) = aStatus;
	return rtOk;
}

/* -------------------------------------------------------------------------- */
int SetPinBehavior(pDevEnt apDevice, int aPinOpt, int aLine)
{
	DEVICEIscPins(apDevice) = aPinOpt;
	return rtOk;
}

/* -------------------------------------------------------------------------- */
// Blank user code
int SetBlankUserCode(pDevEnt apDevice, PSTRING aCode, int aLine)
{
	TSTRING vStr = *aCode;
	STRMove(DEVICEBlankUserCode(apDevice), vStr);
	return rtOk;
}

/* -------------------------------------------------------------------------- */
// Protection
int SetProtectionBits(pDevEnt apDevice, int aRead, int aProg, int aEra,
			int aLine)
{
	DEVICEIscProtect(apDevice)[prtDisRead  ] = cnoBit;
	DEVICEIscProtect(apDevice)[prtDisProg  ] = cnoBit;
	DEVICEIscProtect(apDevice)[prtDisErase ] = cnoBit;

	if (apDevice != NULL)
	{
		DEVICEIscProtect(apDevice)[prtDisRead] = aRead;
		DEVICEIscProtect(apDevice)[prtDisProg] = aProg;
		DEVICEIscProtect(apDevice)[prtDisErase ] = aEra;
    return rtOk;
	}
	return rtErr;
}

/* -------------------------------------------------------------------------- */
int SetProtectionKeyRange(pDevEnt apDevice, int aMsb, int aLsb, int aLine)
{
	DEVICEIscProtect(apDevice)[prtMsb ] = cnoBit;
	DEVICEIscProtect(apDevice)[prtLsb ] = cnoBit;

	if (apDevice != NULL)
	{
		DEVICEIscProtect(apDevice)[prtMsb ] = aMsb;
		DEVICEIscProtect(apDevice)[prtLsb ] = aLsb;
		return rtOk;
	}
 return rtErr;
}

/* -------------------------------------------------------------------------- */
// Flow
int AddFlow(pDevEnt aDevEnt, PSTRING aFlowName, PSTRING aDataName, int aLine)
{ int rt = rtOk;
	pIscFlowEnt vFlow;
  vFlow = InitFlow(aFlowName, aDataName, aLine);
	if (vFlow!=NULL)
		AddListItem(GDvIscFlowList(aDevEnt),vFlow);
	return rt;
}

/* -------------------------------------------------------------------------- */
// . Block
int AddFlowBlock(pDevEnt apDevEnt, int aBlockId, int aLoopCount, int aLine)
{
	TSTRING vString;
	pList vFlowList = GDvIscFlowList(apDevEnt);
	int vCount    = vFlowList->Count;
	pIscFlowEnt vFlow = (pIscFlowEnt)vFlowList->items[vCount-1];

	if (apDevEnt != NULL)
	{
		switch (aBlockId)
		{
			case bkInit:   // Init
				STRInit(vString, "Initialize");
				vFlow->currBlockId = aBlockId;
				vFlow->ifeBlocks[aBlockId]= InitExecBlock(&vString, aLoopCount);
				break;
			case bkRepeat:  // Repeat
				STRInit(vString, "Repeat");
				vFlow->currBlockId = aBlockId;
				vFlow->ifeBlocks[aBlockId]= InitExecBlock(&vString, aLoopCount);
				// Repeat-Block Only check the LoopCount
				if (aLoopCount <=0)
				{
					STRChConcat(apDevEnt->mSemantikErrors, "    ...line ");
					STRIntConcat(apDevEnt->mSemantikErrors, aLine);
					STRChConcat(apDevEnt->mSemantikErrors,
						": loop count must be greater then 0.\n");
					return rtErr;
				}
				break;
			case bkTerminate:  // Done
				STRInit(vString, "Terminate");
				vFlow->currBlockId = aBlockId;
				vFlow->ifeBlocks[aBlockId]= InitExecBlock(&vString, aLoopCount);
				break;
			default:
				return rtErr;
		} // switch aBlockId
    return rtOk;
	}
	return rtErr;
}

/* -------------------------------------------------------------------------- */
// .. Activity
// Add the activity to current block
int AddFlowActivity(pDevEnt aDevEnt, PSTRING aInstructionName, int aLine)
{int rt = rtOk;
 pList vFlowList = GDvIscFlowList(aDevEnt);
 pList vInstList = GDvInstList(aDevEnt);
 int vCount      = vFlowList->Count;
 pIscFlowEnt vFlow	 = (pIscFlowEnt)vFlowList->items[vCount-1];
 pFlowActEnt vAct = InitFlowAct(aInstructionName);
 int vIdx;
 int vBlockId = cnoId;

 if (vFlow !=NULL)  {
   vBlockId = vFlow->currBlockId;
   if (lSearchList(vInstList,aInstructionName,&vIdx)) {
    if (vAct!=NULL) {
      vAct->facInstId = vIdx;
			vAct->facpInst  = (pInstructionEnt)vInstList->items[vIdx];
      vAct->mLine = aLine;
	  AddListItem(&vFlow->ifeBlocks[vBlockId]->exbActList, vAct);
    };
   } else
   {
		rt = rtErr;
		STRChConcat(aDevEnt->mSemantikErrors, "    ...line ");
		STRIntConcat(aDevEnt->mSemantikErrors, aLine);
		STRChConcat(aDevEnt->mSemantikErrors, ": instruction ");
		STRConcat(aDevEnt->mSemantikErrors, *aInstructionName);
		STRChConcat(aDevEnt->mSemantikErrors, " not found.\n");
	 };
 };
 return rt;
};

/* -------------------------------------------------------------------------- */
// ... Update
int AddUpdateField(pDevEnt apDevEnt, int aWidth, pDataExpr apDataExpr,
				int aLine)
{
	pUpFieldEnt vpUpField = InitUpField(aWidth, apDataExpr);
	pList vpActList = GetCurrentActList(apDevEnt);
	pList vpFlowList = GDvIscFlowList(apDevEnt);
	int vActCount = vpActList->Count;
	int vFlowCount = vpFlowList->Count;
	pFlowActEnt vpAct = (pFlowActEnt)vpActList->items[vActCount - 1];
	pIscFlowEnt vpFlow = (pIscFlowEnt)vpFlowList->items[vFlowCount - 1];
	int vIndex;

	if (aWidth <= 0)
	{
		STRChConcat(apDevEnt->mSemantikErrors, "    ...line ");
		STRIntConcat(apDevEnt->mSemantikErrors, aLine);
		STRChConcat(apDevEnt->mSemantikErrors,
			": field width must be greater then 0.\n");
		return rtErr;
	}
	// Add a Variable to Variable Stack
	if (apDataExpr != NULL)
	{
		if (STRPChar(apDataExpr->mVariable) != NULL)
		 //	if (*STRPChar(apDataExpr->mVariable) != "")
      if (STRLen(apDataExpr->mVariable) > 0)
			{
				AddVar(&vpFlow->VarStack, &apDataExpr->mVariable, aWidth, &vIndex);
				vpUpField->DataExpr.mVarEnt
					= (pVarEnt) GetItemAtIndex(&(vpFlow->VarStack), vIndex);
			}
		if (vpAct != NULL)
		{
			AddListItem(&vpAct->UpdateFieldList, vpUpField);
			return rtOk;
		}
	} 
	return rtErr;
}

/* -------------------------------------------------------------------------- */
// ... Wait
int AddWaitDurMin(pDevEnt aDevEnt, pDurationSpec aDurationSpec, int aLine)
{int rt = rtOk;
 pList vActList = GetCurrentActList(aDevEnt);
 int vCount = vActList->Count;
 pFlowActEnt vAct= (pFlowActEnt)vActList->items[vCount-1];
 if (vAct != NULL)
 {
	 vAct->facWaitMin = *aDurationSpec;
 }
	return rt;
}

/* -------------------------------------------------------------------------- */
int AddWaitDurMax(pDevEnt aDevEnt, pDurationSpec aDurationSpec, int aLine)
{int rt = rtOk;
 pList vActList = GetCurrentActList(aDevEnt);
 int vCount = vActList->Count;
 pFlowActEnt vAct= (pFlowActEnt)vActList->items[vCount-1];
 if (vAct != NULL)
	 vAct->facWaitMax = *aDurationSpec;
 return rt;
}

/* -------------------------------------------------------------------------- */
// ... Capture
int AddCaptureField(pDevEnt apDevEnt, int aWidth, int aOutputOp,
			pDataExpr apDataExpr, int aLine)
{
	pCapFieldEnt vpCapField = InitCapField(aWidth, apDataExpr);
	pList vpActList = GetCurrentActList(apDevEnt);
	pList vpFlowList = GDvIscFlowList(apDevEnt);
	int vActCount = vpActList->Count;
	int vFlowCount =  vpFlowList->Count;
	pFlowActEnt vpAct = (pFlowActEnt)vpActList->items[vActCount - 1];
	pIscFlowEnt vpFlow = (pIscFlowEnt)vpFlowList->items[vFlowCount - 1];
	int vIndex;

	if (aWidth <= 0)
	{
		STRChConcat(apDevEnt->mSemantikErrors, "    ...line ");
		STRIntConcat(apDevEnt->mSemantikErrors, aLine);
		STRChConcat(apDevEnt->mSemantikErrors,
			": field width must be greater then 0.\n");
		return rtErr;
	}
	// Add a Variable to Variable Stack
	if (apDataExpr!=NULL)
	{
		if (STRPChar(apDataExpr->mVariable) != NULL)
		 //	if (*STRPChar(apDataExpr->mVariable) != "")
      if (STRLen(apDataExpr->mVariable) > 0)
			{
				AddVar(&vpFlow->VarStack, &apDataExpr->mVariable, aWidth, &vIndex);
				vpCapField->ExpectData.mVarEnt
					= (pVarEnt) GetItemAtIndex(&(vpFlow->VarStack), vIndex);
			}
		if (vpAct != NULL)
		{
			AddListItem(&vpAct->CaptureFieldList, vpCapField);
			return rtOk;
		}
	}
	return rtErr;
}

/* -------------------------------------------------------------------------- */
int SetCRC(pDevEnt aDevEnt, int aCRC, int aLine)
{int rt = rtOk;
 pCapFieldEnt vCapField = GetLastCapField(aDevEnt);
  if (vCapField !=NULL)
		vCapField->CRCtag = aCRC;
	return rt;
};

/* -------------------------------------------------------------------------- */
 pCapFieldEnt GetLastCapField(pDevEnt aDevEnt)
 { pList vActList = GetCurrentActList(aDevEnt);
   int vCount = vActList->Count;
   pFlowActEnt vAct= (pFlowActEnt)vActList->items[vCount-1];
   pCapFieldEnt vCapField;
   vCount = vAct->CaptureFieldList.Count;
   vCapField = (pCapFieldEnt)vAct->CaptureFieldList.items[vCount-1];
	 return(vCapField);
 };

/* -------------------------------------------------------------------------- */
int AddCompareMask(pDevEnt apDevEnt, int aOutputOp, pDataExpr apDataExpr, int aLine)
{int rt = rtOk;
 pCapFieldEnt vpCapField = GetLastCapField(apDevEnt);
 int vIndex;

 pList vpFlowList = GDvIscFlowList(apDevEnt);
	int vFlowCount =  vpFlowList->Count;
	pIscFlowEnt vpFlow = (pIscFlowEnt)vpFlowList->items[vFlowCount - 1];

	if (vpCapField !=NULL)
	{
	    DoneBitField(&(vpCapField->mMaskValue));
		vpCapField->CapOutput =  aOutputOp;
		vpCapField->CompMask  = *apDataExpr;
		STRInit(vpCapField->CompMask.mHexStringOperand, "");
	    if ((apDataExpr->mOpType == opHexStringConstant) || (apDataExpr->mOpType == opAssignment))
	       STRAssign(vpCapField->CompMask.mHexStringOperand, apDataExpr->mHexStringOperand);

		vpCapField->mMask = 1;
        if (apDataExpr->mOpType == opHexStringConstant)
           HexToBitField(&(vpCapField->mMaskValue), &(apDataExpr->mHexStringOperand), vpCapField->Width);		

  	if (STRPChar(apDataExpr->mVariable) != NULL)
	 //		if (*STRPChar(apDataExpr->mVariable) != "")
      if (STRLen(apDataExpr->mVariable) > 0)
			{
				AddVar(&vpFlow->VarStack, &apDataExpr->mVariable, vpCapField->Width, &vIndex);
				vpCapField->CompMask.mVarEnt
					= (pVarEnt) GetItemAtIndex(&(vpFlow->VarStack), vIndex);
			}
	}
	return (rt);
}

/* -------------------------------------------------------------------------- */
// Procedure
int AddProcedure(pDevEnt aDevEnt, PSTRING aProcName, PSTRING aDataName, int aLine)
{int rt = rtOk;
 pProcEnt vProc = InitProc(aProcName,aDataName,aLine);
 if (vProc != NULL)
	 AddListItem(GDvProcList(aDevEnt),vProc);         
	return rt;
}

/* -------------------------------------------------------------------------- */
int AddProcFlow(pDevEnt aDevEnt, PSTRING aFlowName, PSTRING aDataName, int aLine)
{int rt = rtOk;
 int vIdx = cnoId;
 TSTRING vDataName = *aDataName;
 TSTRING vFlowName = *aFlowName;
 pList vProcList = GDvProcList(aDevEnt);
 pList vFlowList = GDvIscFlowList(aDevEnt);
 int   vCount   = vProcList->Count;
 pProcEnt vProc = (pProcEnt)vProcList->items[vCount-1];
 pIscFlowEnt vFlow;
 pIscFlowEnt vCmpFlow;
 if (vProc != NULL) {
   vCmpFlow = InitFlow(aFlowName,aDataName,cnoId);
   if (vCmpFlow != NULL) {
     if (lSearchList(vFlowList,vCmpFlow, &vIdx)) {
       vFlow = (pIscFlowEnt)vFlowList->items[vIdx];
  } else {
    rt = rtErr;
    STRChConcat(aDevEnt->mSemantikErrors, "    ...line ");
			STRIntConcat(aDevEnt->mSemantikErrors, aLine);
			STRChConcat(aDevEnt->mSemantikErrors, ": flow ");
			STRConcat(aDevEnt->mSemantikErrors, vFlowName);
			STRChConcat(aDevEnt->mSemantikErrors, " not found.\n");
			}; // lsearch
   FreeFlow(vCmpFlow);
  }; // if vCmpFlow
 };  // if vProc
	if (rt == 0) AddListItem(&vProc->preFlowList,vFlow);  
	return rt;
}

/* -------------------------------------------------------------------------- */
// Action
int AddAction(pDevEnt aDevEnt, PSTRING aActionName, PSTRING aDataName, int aPROPRIETARY,
			int aLine)
{int rt = rtOk;
 pActionEnt vAct = InitAction(aActionName,aDataName);
 if (vAct !=NULL) {
   vAct->prop = aPROPRIETARY;
   AddListItem(GDvActionList(aDevEnt),vAct);
 };      
	return rt;
}

/* -------------------------------------------------------------------------- */
int AddActionSpec(pDevEnt aDevEnt, PSTRING aProcName, PSTRING aDataName, int aPROPRIETARY,
			int OptRec, int aLine)
{int rt = rtOk;
 int vIdx = cnoId;
 TSTRING vName = *aProcName;
 TSTRING vDataName = *aDataName;
 pActProcEnt ActProc = InitActProc(aProcName,aDataName);
 pList vActList =  GDvActionList(aDevEnt);
 int vCount = vActList->Count;
 pActionEnt vAction = (pActionEnt)vActList->items[vCount-1];
 pList vProcList = GDvProcList(aDevEnt);
 pProcEnt vCmpProc = NULL;
 pProcEnt vProc = NULL;

 if (vProcList !=NULL)
 {
   vCmpProc = InitProc(aProcName,aDataName,cnoId);
	 if (lSearchList(vProcList,vCmpProc, &vIdx))
	 {
    vProc = (pProcEnt)vProcList->items[vIdx];
    ActProc->aseProc = vProc;
    ActProc->options = OptRec;
    ActProc->prop    = aPROPRIETARY;
    AddListItem(&vAction->aceProcList,ActProc);
	 }
	 else
	 {
     rt = rtErr;
     STRChConcat(aDevEnt->mSemantikErrors, "    ...line ");
			STRIntConcat(aDevEnt->mSemantikErrors, aLine);
			STRChConcat(aDevEnt->mSemantikErrors, ": procedure ");
			STRConcat(aDevEnt->mSemantikErrors, vName);
			STRChConcat(aDevEnt->mSemantikErrors, " (");
			STRConcat(aDevEnt->mSemantikErrors, vDataName);
			STRChConcat(aDevEnt->mSemantikErrors, ") not found.\n");
	 };
  FreeProc(vCmpProc);
 };
 return rt;
};

/* -------------------------------------------------------------------------- */
// Illegal exit
int AddExitInstruction(pDevEnt aDevEnt, PSTRING aExitInstr, int aLine)
{
	int rt = rtOk;
	return rt;
}

// **************************************************//
// Functions called from data file parser
// Standard IEEE_1532_2000:

/* -------------------------------------------------------------------------- */
int AddDataRecord(pDevEnt aDevEnt, PSTRING aDataName, int aLine)
{
	int rt = rtOk;

	pDataRecord vpDataRecord = InitDataRecord(aDataName);
	if (vpDataRecord != NULL)
		AddListItem(GDvDataRecordList(aDevEnt), vpDataRecord);
	return rt;
}

/* -------------------------------------------------------------------------- */
//int AddDataBlock(pDevEnt aDevEnt, int aBlockId, TFPOS aFilePos, int aLine)
int AddDataBlock(pDevEnt aDevEnt, int aBlockId, long aFilePos, int aLine)
{
	pList vDataRecordList = GDvDataRecordList(aDevEnt);
	pDataRecord vpDataRecord = NULL;

	int rt = rtErr;
	int vCount = vDataRecordList->Count;

	if (vCount > 0)
		vpDataRecord = (pDataRecord)vDataRecordList->items[vCount -1];

	if (vpDataRecord !=NULL)
	{
		rt = rtOk;
		switch (aBlockId)
		{
			case bkInit:
				vpDataRecord->mInitBlockPos = aFilePos;
				break;
			case bkRepeat:
				vpDataRecord->mRepeatBlockPos = aFilePos;
				break;
			case bkTerminate:
				vpDataRecord->mTerminateBlockPos = aFilePos;
				break;
			default:
				rt = rtErr;
		 }
	}
	return rt;
	return 0;
}

/* -------------------------------------------------------------------------- */
pDataRecord GetDataRecordByName(pActionStatus apStatus, PSTRING aDataName)
{
	int vIdx = cnoId;
	pDataRecord vpDataRecord = NULL;
	pDataRecord vCmpRecord= InitDataRecord(aDataName);
	pList vpDataRecordList = &(apStatus->mDataRecordList);

	if (ITEMCountOfList(vpDataRecordList) == 0)
		return vpDataRecord;
	if (vpDataRecordList->items[vpDataRecordList->Count - 1] != NULL)
		if (vCmpRecord != NULL)
			if (lSearchList(vpDataRecordList, vCmpRecord, &vIdx))
				vpDataRecord = (pDataRecord)vpDataRecordList->items[vIdx];
    FreeDataRecord(vCmpRecord);
//    FREEMem(vCmpRecord, sizeof(tDataRecord));	
	return vpDataRecord;
}


//****************************************************
// KeyOf Functions for BSDLInfo-Lists

/* -------------------------------------------------------------------------- */
void *KeyOfInstList(void *aItem)
{
	pInstructionEnt vInst = (pInstructionEnt) aItem;
	return(&vInst->mInstructionName);
}

/* -------------------------------------------------------------------------- */
void *KeyOfDRList(void *aItem)
{pDataRegEnt vDataReg = (pDataRegEnt) aItem;
	return(&vDataReg->mDataRegName);
}

/* -------------------------------------------------------------------------- */
void *KeyOfFlowList(void *aItem)
{pIscFlowEnt vFlow = (pIscFlowEnt) aItem;
 return(vFlow);
};

/* -------------------------------------------------------------------------- */
void *KeyOfProcList(void *aItem)
{pProcEnt vProc = (pProcEnt) aItem;
 return(vProc);
};

/* -------------------------------------------------------------------------- */
void *KeyOfAction(void *aItem)
{  pActionEnt vAction = (pActionEnt) aItem;
   return(vAction);
};

/* -------------------------------------------------------------------------- */
void *KeyOfVarStack(void *aItem)
{ pVarEnt aVar = (pVarEnt) aItem;
  return(&aVar->varName);
};

/* -------------------------------------------------------------------------- */
void *KeyOfDataRecordList(void *aItem)
{
	pDataRecord vpDataRecord = (pDataRecord) aItem;
	return(&vpDataRecord->mDataName);
}

// ****************************************************
// PSTRING Compareration

/* -------------------------------------------------------------------------- */
int CompareIscFlow(void *aFlow1, void *aFlow2)
{int rt = 0;
 pIscFlowEnt vFlow1 = (pIscFlowEnt) aFlow1;
 pIscFlowEnt vFlow2 = (pIscFlowEnt) aFlow2;
 rt = STRCompare(vFlow1->ifeFlowName,vFlow2->ifeFlowName);
 if (rt == 0) {
   rt = STRCompare(vFlow1->ifeDataName,vFlow2->ifeDataName);
 };
 return(rt);
};

/* -------------------------------------------------------------------------- */
int CompareProc(void *aProc1, void *aProc2)
{ int rt = 0;
  pProcEnt vProc1 = (pProcEnt) aProc1;
  pProcEnt vProc2 = (pProcEnt) aProc2;
  rt = STRCompare(vProc1->preName,vProc2->preName);
  if (rt == 0) {
    rt = STRCompare(vProc1->preDataName,vProc2->preDataName);
  };
 return(rt);
};

/* -------------------------------------------------------------------------- */
int ComparepString(void * aStr1, void *aStr2)
{ //PSTRING Str1 = (PSTRING *)aStr1;
  //PSTRING Str2 = (PSTRING *)aStr2;
  PSTRING Str1 = (PSTRING)aStr1;
  PSTRING Str2 = (PSTRING)aStr2;
  TSTRING tStr1 = *Str1;
  TSTRING tStr2 = *Str2;
  return(STRCompare(tStr1,tStr2));
}

/* -------------------------------------------------------------------------- */
int CompareAction(void *aAction1, void *aAction2)
{int rt = 0;
  pActionEnt vAction1 = (pActionEnt) aAction1;
  pActionEnt vAction2 = (pActionEnt) aAction2;
  rt = STRCompare(vAction1->aceName,vAction2->aceName);
	if (rt == 0) {
    rt = STRCompare(vAction1->aceDataName,vAction2->aceDataName);
  };
 return(rt);
};

/* -------------------------------------------------------------------------- */
pExecBlock GetBlockAtIndex(pIscFlowEnt aFlow, int aIndex)
{pExecBlock aBlock = NULL;
  if (aFlow!=NULL) {
    if ((aIndex >=0) && (aIndex < 3))
      aBlock = aFlow->ifeBlocks[aIndex];
  };
	return(aBlock);
}

/* -------------------------------------------------------------------------- */
pActionEnt GetActionByName(pDevEnt aDevEnt,PSTRING aActionName, PSTRING aDataName)
{
	int vIdx = cnoId;
	pActionEnt vAction = NULL;
	pList vActionList = GDvActionList(aDevEnt);


	pActionEnt vCmpAction = InitAction(aActionName, aDataName);
	if (vActionList->items[vActionList->Count - 1] != NULL)
		if (vCmpAction != NULL)
			if (lSearchList(vActionList, vCmpAction, &vIdx))
				vAction = (pActionEnt)vActionList->items[vIdx];
    FreeAction(vCmpAction);
	return vAction;
}

/* -------------------------------------------------------------------------- */
pProcEnt GetProcedureByName(pDevEnt apDevice, PSTRING aProcName)
{
	int vIdx;
	pList vpProcedureList = GDvProcList(apDevice);
	pProcEnt vpProcedure = NULL;
	pProcEnt vpCmpProc;

	TSTRING vDataName;
	STRInit(vDataName, "");

	vpCmpProc = InitProc(aProcName, &vDataName , cnoId);
 	if (apDevice != NULL)
		if (lSearchList(vpProcedureList, vpCmpProc, &vIdx))
			vpProcedure = (pProcEnt)GetItemAtIndex(vpProcedureList, vIdx);
	return vpProcedure;
}

/* -------------------------------------------------------------------------- */
int GetBypassInstructionCodeOfDevice(pDevEnt apDevice, PSTRING aCode)
{
	pInstructionEnt vpInstruction;
	TSTRING vInstructionName;
	int i;

 	if (apDevice != NULL)
	{
	    STRInit(vInstructionName, "BYPASS");
		vpInstruction = GetInstructionByName(apDevice, &vInstructionName);
		if (vpInstruction != NULL)
			STRAssign(*aCode, vpInstruction->mInstructionOpCode);
		else
			for (i = 0; i < IRLengthOfDevice(apDevice); i++)
				STRAddChar(*aCode, '1');
        STRDone(vInstructionName);
		return rtOk;
	}
	return rtErr;
}
/* Frank Phillips @Xilinx Added the following two functions on
May_01_2001
*/
/* -------------------------------------------------------------------------- */
int GetEnableInstructionCodeOfDevice(pDevEnt apDevice, PSTRING aCode)
{
    pInstructionEnt vpInstruction;
    TSTRING vInstructionName;
    int i;

    STRInit(vInstructionName, "ISC_ENABLE");
    if (apDevice != NULL)
    {
        vpInstruction = GetInstructionByName(apDevice, &vInstructionName);
        if (vpInstruction != NULL)
            STRAssign(*aCode, vpInstruction->mInstructionOpCode);
        else
            for (i = 0; i < IRLengthOfDevice(apDevice); i++)
                STRAddChar(*aCode, '1');
        return rtOk;
    }
    return rtErr;
}
/* -------------------------------------------------------------------------- */
int GetDisableInstructionCodeOfDevice(pDevEnt apDevice, PSTRING aCode)
{
    pInstructionEnt vpInstruction;
    TSTRING vInstructionName;
    int i;

    STRInit(vInstructionName, "ISC_DISABLE");
    if (apDevice != NULL)
    {
        vpInstruction = GetInstructionByName(apDevice, &vInstructionName);
        if (vpInstruction != NULL)
            STRAssign(*aCode, vpInstruction->mInstructionOpCode);
        else
            for (i = 0; i < IRLengthOfDevice(apDevice); i++)
                STRAddChar(*aCode, '1');
        return rtOk;
    }
    return rtErr;
}
/* -------------------------------------------------------------------------- */
pInstructionEnt GetInstructionByName(pDevEnt apDevice, PSTRING aInstructionName)
{
	int vIdx = 0;
	pInstructionEnt vpInstruction = NULL;
	pList pInstructionList = GDvInstList(apDevice);

	if (apDevice != NULL)
		if (lSearchList(pInstructionList, aInstructionName, &vIdx))
			vpInstruction = (pInstructionEnt)pInstructionList->items[vIdx];

	return vpInstruction;
}

/* -------------------------------------------------------------------------- */
int BsdlSemanticCheck(pDevEnt apDevice)
{
	int i, j, k, l;
	int vUpdateFieldSum, vCaptureFieldSum;
	pList vpFlowList, vpActivityList, vpUpdateFieldList, vpCaptureFieldList;
	pIscFlowEnt vpFlow;
	pExecBlock vpBlock;
	pFlowActEnt vpActivity;
	pUpFieldEnt vpUpdateField;
	pCapFieldEnt vpCaptureField;

	// get bsdl file name
	char *vFileName = STRPChar(BSDLFileNameOfDevice(apDevice));
	if (vFileName == NULL)
		return rtOk;

	// compare field with summaries
	vpFlowList = GDvIscFlowList(apDevice);
	for (i = 0; i < ITEMCountOfList(vpFlowList); i++)
	{
		vpFlow = (pIscFlowEnt) GetItemAtIndex(vpFlowList, i);
		for (j = 0; j < 3; j++)
		{
			vpBlock = GetBlockAtIndex(vpFlow, j);
			if (vpBlock != NULL)
			{
				vpActivityList = ACTIVITYListOfBlock(vpBlock);
				if (vpActivityList != NULL)
					for (k = 0; k < ITEMCountOfList(vpActivityList); k++)
					{
						vpActivity = (pFlowActEnt) GetItemAtIndex(vpActivityList, k);
						// field width summary of update fields
						vpUpdateFieldList = &vpActivity->UpdateFieldList;
						vUpdateFieldSum = 0;
						for (l = 0; l < ITEMCountOfList(vpUpdateFieldList); l++)
						{
							vpUpdateField
								= (pUpFieldEnt) GetItemAtIndex(vpUpdateFieldList, l);
							vUpdateFieldSum += vpUpdateField->Width;
						}
						vpActivity->mUpdateFieldSum = vUpdateFieldSum;
						// field width summary of capture fields
						vpCaptureFieldList = &vpActivity->CaptureFieldList;
						vCaptureFieldSum = 0;
						for (l = 0; l < ITEMCountOfList(vpCaptureFieldList); l++)
						{
							vpCaptureField
								= (pCapFieldEnt) GetItemAtIndex(vpCaptureFieldList, l);
							vCaptureFieldSum += vpCaptureField->Width;
						}
						vpActivity->mCaptureFieldSum = vCaptureFieldSum;
						// compare update and capture field width summaries
						if ((vUpdateFieldSum > 0) && (vCaptureFieldSum > 0))
						{
							if (vUpdateFieldSum != vCaptureFieldSum)
							{
								STRChConcat(apDevice->mSemantikErrors, "    ...line ");
								STRIntConcat(apDevice->mSemantikErrors, vpActivity->mLine);
								STRChConcat(apDevice->mSemantikErrors,
									": update field width sum != capture field width sum.\n");
							}
						}
					}	// end activity loop
			}
		}	// end block loop
	}	// end flow loop
	if (STRLen(apDevice->mSemantikErrors) > 0)
	{
		WRITEError("  ...Semantic error(s):\n", vFileName);
		WRITEError(STRPChar(apDevice->mSemantikErrors));
		STRDone(apDevice->mSemantikErrors);
		return rtErr;
	}
	return rtOk;
}

/* -------------------------------------------------------------------------- */
int DataSemanticCheck(pDevEnt apDevice)
{
	int i, j, k, l;
	int vInputOp, vError;
	pList vpFlowList, vpActivityList, vpUpdateFieldList, vpCaptureFieldList;
	pIscFlowEnt vpFlow;
  pExecBlock vpBlock;
	pFlowActEnt vpActivity;
	pUpFieldEnt vpUpdateField;
	pCapFieldEnt vpCaptureField;
	pDataRecord vpDataRecord;
	char *vFileName;

	// check whether data input file is given
	if (STRLen(DATAInputFileNameOfDevice(apDevice)) == 0)
		return rtOk;
	vFileName = STRPChar(DATAInputFileNameOfDevice(apDevice));
	if (vFileName == NULL)
		return rtOk;

	STRInit(apDevice->mSemantikErrors, "");

	// compare field with summaries
	vpFlowList = GDvIscFlowList(apDevice);
	for (i = 0; i < ITEMCountOfList(vpFlowList); i++)
	{
		vpFlow = (pIscFlowEnt) GetItemAtIndex(vpFlowList, i);
		for (j = 0; j < 3; j++)
		{
    	// check data records
			if (STRLen(vpFlow->ifeDataName) > 0)
			{

				vpDataRecord	= GetDataRecordByName(ACTIONStatusOfDevice(apDevice),
					&(vpFlow->ifeDataName));

				vpBlock = GetBlockAtIndex(vpFlow, j);
				if (vpBlock != NULL)
				{
					vpActivityList = ACTIVITYListOfBlock(vpBlock);
					if (vpActivityList != NULL)
						for (k = 0; k < ITEMCountOfList(vpActivityList); k++)
						{
							vpActivity = (pFlowActEnt) GetItemAtIndex(vpActivityList, k);

							// check update fields for input ops
							vInputOp = 0;
							vpUpdateFieldList = &vpActivity->UpdateFieldList;
							for (l = 0; l < ITEMCountOfList(vpUpdateFieldList); l++)
							{
								vpUpdateField
									= (pUpFieldEnt) GetItemAtIndex(vpUpdateFieldList, l);
								if (vpUpdateField->DataExpr.mOpType == opInput)
									vInputOp = 1;
							}
              // check capture fields for input ops
							vpCaptureFieldList = &vpActivity->CaptureFieldList;
							for (l = 0; l < ITEMCountOfList(vpCaptureFieldList); l++)
							{
								vpCaptureField
									= (pCapFieldEnt) GetItemAtIndex(vpCaptureFieldList, l);
								if ((vpCaptureField->ExpectData.mOpType == opInput)
										|| (vpCaptureField->CompMask.mOpType == opInput))
									vInputOp = 1;
							}
						} // end activity loop

						// if input operator used in activity, check whether corresponding
						// data section exists in data input file
						if (vInputOp)
						{
							// data section not defined
							if (vpDataRecord == NULL)
							{
								STRChConcat(apDevice->mSemantikErrors,
									"    ...Data section for flow \"");
								STRConcat(apDevice->mSemantikErrors, vpFlow->ifeFlowName);
								STRChConcat(apDevice->mSemantikErrors, " (");
								STRConcat(apDevice->mSemantikErrors, vpFlow->ifeDataName);
								STRChConcat(apDevice->mSemantikErrors, ")\" not found.\n");
								// continue flow loop
								j = 3;
							}
							// block in data section not defined
							else
							{
								vError = 0;
								switch (j)
								{
									case 0:
										if (vpDataRecord->mInitBlockPos == -1)
										{
											STRChConcat(apDevice->mSemantikErrors,
												"    ...Initialize block in data section for flow \"");
											vError = 1;
										}
										break;
									case 1:
										if (vpDataRecord->mRepeatBlockPos == -1)
										{
											STRChConcat(apDevice->mSemantikErrors,
												"    ...Repeat block in data section for flow \"");
											vError = 1;
										}
										break;
									case 2:
										if (vpDataRecord->mTerminateBlockPos == -1)
										{
											STRChConcat(apDevice->mSemantikErrors,
												"    ...Terminate block in data section for flow \"");
											vError = 1;
										}
										break;
								}
								if (vError)
								{
									STRConcat(apDevice->mSemantikErrors, vpFlow->ifeFlowName);
									STRChConcat(apDevice->mSemantikErrors, " (");
									STRConcat(apDevice->mSemantikErrors, vpFlow->ifeDataName);
									STRChConcat(apDevice->mSemantikErrors, ")\" not found.\n");
								}
							}
						} // if vInputOp
				}
			}
		} // end block loop
	} // end flow loop

	// write out errors
	if (STRLen(apDevice->mSemantikErrors) > 0)
	{
		WRITEError("  ...Semantic error(s):\n", vFileName);
		WRITEError(STRPChar(apDevice->mSemantikErrors));
		STRDone(apDevice->mSemantikErrors);
		return rtErr;
	}
	return rtOk;
}





